﻿using System.Data;
using FujiSDK;
using Metal;
using CCameraControl;

namespace ZoomPos;

public partial class ZoomPosForm : ContentPage
{
	private string state = "DisConnect";
	private ZoomPosControl zoomPosCameraControl;
	private IDispatcherTimer timer;
	private Label ConnectState = new Label { Text = " Waiting for a camera to be connected.", FontSize = 18, TextColor = Colors.Black };
	private BoxView ContentDivider = new BoxView{ Color = Colors.Black, HeightRequest = 1, HorizontalOptions = LayoutOptions.Fill };
	private Label ZoomFocalLengthLabel = new Label { Text = "Zoom focal length:", FontSize = 18, TextColor = Colors.Black };
	private Label ZoomFocalLength = new Label { Text = "- - mm", FontSize = 18, HorizontalTextAlignment = TextAlignment.End, TextColor = Colors.Black };
	private Label ZoomFocalLength35Label = new Label { Text = "Zoom focal length (in 35mm format):", FontSize = 18, TextColor = Colors.Black };
	private Label ZoomFocalLength35 = new Label { Text = "- - mm", FontSize = 18, HorizontalTextAlignment = TextAlignment.End, TextColor = Colors.Black };
	private Label BusyCountLabel = new Label { Text = "Communication BUSY count:", FontSize = 18, TextColor = Colors.Black };
	private Label BusyCount = new Label { Text = "0", FontSize = 18, HorizontalTextAlignment = TextAlignment.End, TextColor = Colors.Black };
	private Button ExitButton = new Button { Text = "Exit", TextColor = Colors.Black, BackgroundColor = Colors.White, BorderColor = Colors.Blue, BorderWidth = 1, FontSize = 18 };

	public ZoomPosForm()
	{
		InitializeComponent();
		CreateView();
	}

	protected override void OnAppearing()
	{
		base.OnAppearing();
		zoomPosCameraControl = new ZoomPosControl();

		SetTimer();

		return;
	}
	// Exit Application
	protected override void OnDisappearing()
	{
		base.OnDisappearing();
		if (timer != null)
		{
			timer.Stop();
			timer = null;
		}
		zoomPosCameraControl.TerminateConnection();	
	}

	private async void SetTimer()
	{
		bool bLoad = zoomPosCameraControl.LoadLibrary();
		if (!bLoad)
		{
			// err
			await DisplayAlert("Error","Failed to load XAPI.bundle", "OK");
			Environment.Exit(0);
			return;
		}

		zoomPosCameraControl.SetDetectOption(CameraControl.XSDK_DSC_IF_USB, null);
		zoomPosCameraControl.BeginConnectThread();

		timer = Application.Current.Dispatcher.CreateTimer();
		timer.Interval = TimeSpan.FromMilliseconds(500);
		timer.IsRepeating = true;
		timer.Tick += (s, e) => ConnectCheck();
		timer.Start();
	}

	private void ConnectCheck()
	{
		switch (state)
		{
			case "DisConnect":
				if (zoomPosCameraControl != null)
				{
					// Connection Check
					if (zoomPosCameraControl.IsConnectThread() == ZoomPosControl.FALSE)
					{
						zoomPosCameraControl.EndConnectThread();
						zoomPosCameraControl.BeginPollThread();

						state = "Connect";
						Task.CompletedTask.ContinueWith(t =>
						{
							ConnectState.Text = " Connect";
						},TaskScheduler.FromCurrentSynchronizationContext());
						ConnectState.Text = " Connect";
					}
				}
				break;
			case "Connect":
				if (zoomPosCameraControl != null)
				{
					// Connection Check
					if (zoomPosCameraControl.IsPollThread() != ZoomPosControl.FALSE)
					{
						long lFocalLength;
						long l35mmFocalLength;
						int busyCount;

						lFocalLength = zoomPosCameraControl.GetFocalLength();
						if (lFocalLength > 0)
						{
							Task.CompletedTask.ContinueWith(t =>
							{
								ZoomFocalLength.Text = (lFocalLength / 100).ToString() + "." + (lFocalLength % 100).ToString() + " mm";
							},TaskScheduler.FromCurrentSynchronizationContext());
							ZoomFocalLength.Text = (lFocalLength / 100).ToString() + "." + (lFocalLength % 100).ToString() + " mm";
						}
						else
						{
							Task.CompletedTask.ContinueWith(t =>
							{
								ZoomFocalLength.Text = "- - mm";
							},TaskScheduler.FromCurrentSynchronizationContext());
							ZoomFocalLength.Text = "- - mm";
						}

						l35mmFocalLength = zoomPosCameraControl.Get35mmFocalLength();
						if (l35mmFocalLength > 0)
						{
							Task.CompletedTask.ContinueWith(t =>
							{
								ZoomFocalLength35.Text = (l35mmFocalLength / 100).ToString() + "." + (l35mmFocalLength % 100).ToString() + " mm";
							},TaskScheduler.FromCurrentSynchronizationContext());
							ZoomFocalLength35.Text = (l35mmFocalLength / 100).ToString() + "." + (l35mmFocalLength % 100).ToString() + " mm";
						}
						else
						{
							Task.CompletedTask.ContinueWith(t =>
							{
								ZoomFocalLength35.Text = "- - mm";
							},TaskScheduler.FromCurrentSynchronizationContext());
							ZoomFocalLength35.Text = "- - mm";
						}

						busyCount = zoomPosCameraControl.GetBusyCount();
						Task.CompletedTask.ContinueWith(t =>
						{
							BusyCount.Text = busyCount.ToString();							
						},TaskScheduler.FromCurrentSynchronizationContext());
						BusyCount.Text = busyCount.ToString();
					}
					else
					{
						zoomPosCameraControl.EndPollThread();
						zoomPosCameraControl.TerminateConnection();
						zoomPosCameraControl.SetDetectOption(CameraControl.XSDK_DSC_IF_USB, null);
						zoomPosCameraControl.BeginConnectThread();

						state = "DisConnect";
						Task.CompletedTask.ContinueWith(t =>
						{
							ConnectState.Text = " Waiting for a camera to be connected.";
							ZoomFocalLength.Text = "- - mm";
							ZoomFocalLength35.Text = "- - mm";
							BusyCount.Text = "0";
						},TaskScheduler.FromCurrentSynchronizationContext());
						ConnectState.Text = " Waiting for a camera to be connected.";
						ZoomFocalLength.Text = "- - mm";
						ZoomFocalLength35.Text = "- - mm";
						BusyCount.Text = "0";
					}
				}
				break;
		}
	}
	
	// Exit Application
	public void OnClickExit(object sender, EventArgs e)
	{
		App.Current?.Quit();
	}

	private void CreateView()
	{
		ExitButton.Clicked += OnClickExit;
		
		Frame frame = new Frame{ Content = ConnectState, BorderColor = Colors.Black, BackgroundColor = Colors.LightGray, Padding = new Thickness(2), CornerRadius = 0};
		
		AbsoluteLayout.SetLayoutBounds(frame, new Rect(30, 15, AbsoluteLayout.AutoSize, AbsoluteLayout.AutoSize));
		AbsoluteLayout.SetLayoutBounds(ContentDivider, new Rect(0, 55, 10000, 1));
		AbsoluteLayout.SetLayoutBounds(ZoomFocalLengthLabel, new Rect(30, 70, 350, 22));
		AbsoluteLayout.SetLayoutBounds(ZoomFocalLength, new Rect(270, 110, 180, 22));
		AbsoluteLayout.SetLayoutBounds(ZoomFocalLength35Label, new Rect(30, 150, 350, 22));
		AbsoluteLayout.SetLayoutBounds(ZoomFocalLength35, new Rect(270, 180, 180, 22));
		AbsoluteLayout.SetLayoutBounds(BusyCountLabel, new Rect(30, 240, 350, 22));
		AbsoluteLayout.SetLayoutBounds(BusyCount, new Rect(270, 270, 180, 22));
		AbsoluteLayout.SetLayoutBounds(ExitButton, new Rect(330, 310, 120, 30));

		AbsoluteLayout absoluteLayout = new AbsoluteLayout
		{
			Margin = new Thickness(0),
			Children = {ConnectState, frame, ContentDivider, ZoomFocalLengthLabel, ZoomFocalLength, ZoomFocalLength35Label, ZoomFocalLength35, BusyCountLabel, BusyCount, ExitButton}
		};

		Content = absoluteLayout;
	}
}

