#include <iostream>
#include <string>
#include <vector>
#include <cstring>
#include <algorithm>
#include <iomanip>
#include <limits>
#include "CameraControl.h"

struct CameraInfo 
{
	int index;
	char vendor[256];
	char product[256];
	char firmware[256];
	long sensitivity;
	XSDK_HANDLE hCamera;
};

void DetectedListView(std::vector<XSDK_CameraList> list);
std::vector<int> GetOpenList(std::vector<XSDK_CameraList> detectedList, std::vector<CameraInfo> connectedList);
void OpenListView(std::vector<XSDK_CameraList> detectedList, std::vector<int> openList);
void ConnectedListView(std::vector<CameraInfo> list);
void ErrorView(CCameraControl* cameraControl);
void SelectErrorView();
bool CompareByIndex(const CameraInfo& info1, const CameraInfo& info2);

int main()
{
	bool isToolLoop = true;
	bool isDetected = false;
	bool isConnected = false;
	long result = 0;
	long interface = 0;
	long cameraCount = 0;
	int inputNum = 0;
	int inputIndex = 0;
	std::vector<XSDK_CameraList> detectedList;
	std::vector<int> openList;
	std::vector<CameraInfo> connectedList;
	XSDK_HANDLE hCamera = 0;
	CCameraControl* cameraControl = new CCameraControl();

	bool isLoad = cameraControl->LoadLibraty(); // Loda XAPI.so.
	if(!isLoad)
	{
		// error
		std::cout << "Failed to load XAPI.so" << std::endl;
		return 0;
	}

	result = cameraControl->Init(); // Initialize SDK.
	if(result != XSDK_COMPLETE)
	{
		// error
		ErrorView(cameraControl);
		return 0;
	}

	while(isToolLoop)
	{
		std::cout << std::endl;
		std::cout << "--- Command MENU ---" << std::endl;
		if(!isConnected)
		{
			std::cout << "1.Detect" << std::endl;
		}
		if(isDetected)
		{
			std::cout << "2.Append" << std::endl;
			openList.clear();
			openList = GetOpenList(detectedList, connectedList);
			if(openList.size() != 0) std::cout << "3.Open" << std::endl;
		}
		if(isConnected)
		{
			std::cout << "4.Close" << std::endl;
			std::cout << "5.GetSensitivity" << std::endl;
		}
		if(isDetected)
		{
			std::cout << "6.Show list" << std::endl;
		}
		if(!isConnected)
		{
			std::cout << "7.Exit" << std::endl;
		}
			std::cout << "select number:";
			std::cin >> inputNum;

		switch(inputNum)
		{
			case 1: // Detect select.
			{
				if(isConnected)
				{
					// error.
					SelectErrorView();
					continue;
				}
				std::cout << std::endl;
				std::cout << "--- Detect Interface ---" << std::endl;
				std::cout << "1.USB" << std::endl;
				std::cout << "2.IP(Local network)" << std::endl;
				std::cout << "3.USB and IP(Local network)" << std::endl;
				std::cout << "select number:";
				std::cin >> inputNum;

				switch(inputNum)
				{
					case 1:
						interface = XSDK_DSC_IF_USB;
						break;
					case 2:
						interface = XSDK_DSC_IF_WIFI_LOCAL;
						break;
					case 3:
						interface = XSDK_DSC_IF_USB | XSDK_DSC_IF_WIFI_LOCAL;
						break;
					default:
						SelectErrorView();
						continue;
				}

				detectedList.clear();
				result = cameraControl->Detect(interface, "", "", &cameraCount); // Camera search.
				if(result != XSDK_COMPLETE)
				{
					// error.
					ErrorView(cameraControl);
					continue;
				}
				std::cout << std::endl;
				std::cout << " -- Detect cameras count -- " << std::endl;
				std::cout << "# of cameras:" << cameraCount << std::endl;

				if(cameraCount == 0)
				{
					isDetected = false;
				}
				else
				{
					isDetected = true;
				}

				for(int i = 0; i < cameraCount; i++) // Update list.
				{
					XSDK_CameraList list = {"", "", "", "", true};
					detectedList.push_back(list);
				}
				DetectedListView(detectedList);
			}
				break;
			case 2: // Append select.
			{
				if(!isDetected)
				{
					// error.
					SelectErrorView();
					continue;
				}
				std::cout << std::endl;
				std::cout << "--- Append Interface ---" << std::endl;
				std::cout << "1.USB" << std::endl;
				std::cout << "2.IP(Local network)" << std::endl;
				std::cout << "3.USB and IP(Local network)" << std::endl;
				std::cout << "select number:";
				std::cin >> inputNum;

				switch(inputNum)
				{
					case 1:
						interface = XSDK_DSC_IF_USB;
						break;
					case 2:
						interface = XSDK_DSC_IF_WIFI_LOCAL;
						break;
					case 3:
						interface = XSDK_DSC_IF_USB | XSDK_DSC_IF_WIFI_LOCAL;
						break;
					default:
						SelectErrorView();
						continue;
				}

				detectedList.clear();
				cameraCount = 128;
				XSDK_CameraList* cameraList = new XSDK_CameraList[cameraCount];

				result = cameraControl->Append(interface, "", "", &cameraCount, cameraList); // Camera list acquisition.
				if(result != XSDK_COMPLETE)
				{
					ErrorView(cameraControl);
					delete[] cameraList;
					continue;
				}
				std::cout << std::endl;
				std::cout << " -- Append cameras count -- " << std::endl;
				std::cout << "# of cameras:" << cameraCount << std::endl;

				for(int i = 0; i < cameraCount; i++) // Update list.
				{
					detectedList.push_back(cameraList[i]);
				}
				DetectedListView(detectedList);
				delete[] cameraList;
			}
				break;
			case 3: // Open select.
			{
				if(!isDetected || openList.size() == 0)
				{
					// error.
					SelectErrorView();
					continue;
				}
				std::cout << std::endl;
				std::cout << "--- Open Camera Index ---" << std::endl;
				OpenListView(detectedList, openList);

				std::cout << "select number:";
				std::cin >> inputIndex;

				bool isFound = false;
				for(int i = 0; i < openList.size(); i++)
				{
					if(openList[i] == inputIndex)
					{
						isFound = true;
						break;
					}
				}
				if(!isFound)
				{
					// error.
					SelectErrorView();
					continue;
				}

				std::string deviceStr = "ENUM:";
				std::string indexStr = std::to_string(inputIndex);
				deviceStr = deviceStr + indexStr;
				char device[deviceStr.size()];
				strcpy(device, deviceStr.c_str());
				long cameraMode = 0;

				result = cameraControl->Open(device, &hCamera, &cameraMode, NULL); // Establish a session betwwen the camera.
				if(result != XSDK_COMPLETE)
				{
					// error.
					ErrorView(cameraControl);
					continue;
				}

				XSDK_DeviceInformation deviceInfo;
				result = cameraControl->GetDeviceInfo(hCamera, &deviceInfo); // Get information about the connect camera.
				if(result != XSDK_COMPLETE)
				{
					// error.
					ErrorView(cameraControl);
					continue;
				}

				// Update list.
				CameraInfo info;
				info.index = inputIndex;
				strcpy(info.vendor, deviceInfo.strVendor);
				strcpy(info.product, deviceInfo.strProduct);
				strcpy(info.firmware, deviceInfo.strFirmware);
				info.sensitivity = 0;
				info.hCamera = hCamera;
				connectedList.push_back(info);
				ConnectedListView(connectedList);

				if(connectedList.size() == 0)
				{
					isConnected = false;
				}
				else
				{
					isConnected = true;
				}
			}
				break;
			case 4: // Close select.
			{
				if(!isConnected)
				{
					// error.
					SelectErrorView();
					continue;
				}
				std::cout << std::endl;
				std::cout << "--- Close Camera Index ---" << std::endl;
				ConnectedListView(connectedList);
				std::cout << "select number:";
				std::cin >> inputIndex;

				hCamera = NULL;
				int index = 0;
				for(index; index < connectedList.size(); index++)
				{
					if(connectedList[index].index == inputIndex)
					{
						hCamera = connectedList[index].hCamera;
						break;
					}
				}

				if(hCamera == NULL)
				{
					// error.
					SelectErrorView();
					continue;
				}

				result = cameraControl->Close(hCamera); // Disestablish a session between the camera.
				if(result != XSDK_COMPLETE)
				{
					// error
					ErrorView(cameraControl);
				}

				// Update list.
				connectedList.erase(connectedList.begin() + index);
				ConnectedListView(connectedList);

				if(connectedList.size() == 0)
				{
					isConnected = false;
				}
				else
				{
					isConnected = true;
				}
			}
				break;
			case 5: // Getsensitivity select.
			{
				if(!isConnected)
				{
					// error.
					SelectErrorView();
					continue;
				}
				std::cout << std::endl;
				std::cout << "--- GetSensitivity Camera Index ---" << std::endl;
				ConnectedListView(connectedList);
				std::cout << "select number:";
				std::cin >> inputIndex;

				hCamera = NULL;
				int index = 0;
				for(index; index < connectedList.size(); index++)
				{
					if(connectedList[index].index == inputIndex)
					{
						hCamera = connectedList[index].hCamera;
						break;
					}
				}

				if(hCamera == NULL)
				{
					// error.
					SelectErrorView();
					continue;
				}

				long sensitivity = 0;
				result = cameraControl->GetSensitivity(hCamera, &sensitivity); // Get the ISO sensitivity.
				if(result != XSDK_COMPLETE)
				{
					// error.
					ErrorView(cameraControl);
					continue;
				}
				
				// Update list.
				connectedList[index].sensitivity = sensitivity;
				ConnectedListView(connectedList);
			}
				break;
			case 6: // Show list select.
			{
				if(!isDetected)
				{
					// error.
					SelectErrorView();
					continue;
				}
				DetectedListView(detectedList);
				ConnectedListView(connectedList);
			}
				break;
			case 7: // Exit select
			{
				if(isConnected)
				{
					// error.
					SelectErrorView();
					continue;
				}
				isToolLoop = false;
			}
				break;
			default:
			{
				SelectErrorView();
			}
		}
	}

	result = cameraControl->Exit(); // Termination SDK.
	if(result != XSDK_COMPLETE)
	{
		// error.
		ErrorView(cameraControl);
	}
	
	if(cameraControl)
	{
		delete cameraControl;
		cameraControl = NULL;
	}
	return 0;
}

void DetectedListView(std::vector<XSDK_CameraList> list)
{
	std::cout << std::endl;
	std::cout << " -- Detected cameras: -- " << std::endl;
	std::cout << "index     bValid     strProduct     strSerialNo/strIPAddress" << std::endl;

	for(int i = 0; i < list.size(); i++)
	{
		std::cout << std::left;
		std::cout << std::setw(10) << i;
		if(list[i].bValid == true)
		{
			std::cout << "Valid      ";
		}
		else
		{
			std::cout << "Invalid    ";
		}
		std::cout << std::setw(15) << list[i].strProduct;
		if(strcmp(list[i].strFramework, "USB") == 0)
		{
			std::cout << list[i].strSerialNo << std::endl;
		}
		else
		{
			std::cout << list[i].strIPAddress << std::endl;
		}
	}
}

std::vector<int> GetOpenList(std::vector<XSDK_CameraList> detectedList, std::vector<CameraInfo> connectedList)
{
	std::vector<int> openList;
	for(int i = 0; i < detectedList.size(); i++)
	{
		if(detectedList[i].bValid == false) continue;
		bool isOpen = false;
		for(int j = 0; j < connectedList.size(); j++)
		{
			if(connectedList[j].index == i)
			{
				isOpen = true;
				break;
			}
		}
		if(isOpen) continue;

		openList.push_back(i);
	}

	return openList;
}

void OpenListView(std::vector<XSDK_CameraList> detectedList, std::vector<int> openList)
{
	std::cout << std::endl;
	std::cout << " -- Detected cameras: -- " << std::endl;
	std::cout << "index     bValid     strProduct     strSerialNo/strIPAddress" << std::endl;

	for(int i = 0; i < openList.size(); i++)
	{
		std::cout << std::left;
		std::cout << std::setw(10) << openList[i];
		if(detectedList[openList[i]].bValid == true)
		{
			std::cout << "Valid      ";
		}
		else
		{
			std::cout << "Invalid    ";
		}
		std::cout << std::setw(15) << detectedList[openList[i]].strProduct;
		if(strcmp(detectedList[openList[i]].strFramework, "USB") == 0)
		{
			std::cout << detectedList[openList[i]].strSerialNo << std::endl;
		}
		else
		{
			std::cout << detectedList[openList[i]].strIPAddress << std::endl;
		}
	}
}

void ConnectedListView(std::vector<CameraInfo> list)
{
	std::sort(list.begin(), list.end(), CompareByIndex);

	std::cout << std::endl;
	std::cout << " -- Opened devices: -- " << std::endl;
	std::cout << "index     strVendor     strProduct     strFirmware     Sensitivity" << std::endl;

	for(int i = 0; i < list.size(); i++)
	{
		std::cout << std::setw(10) << list[i].index;
		std::cout << std::setw(14) << list[i].vendor;
		std::cout << std::setw(15) << list[i].product;
		std::cout << std::setw(16) << list[i].firmware;
		std::cout << sensitivityToString(list[i].sensitivity) << std::endl;
	}
}

void ErrorView(CCameraControl* cameraControl)
{
	std::cout << std::endl;
	std::cout << "ERROR" << std::endl;
	std::cout << "apiCode:" << apiCodeToString(cameraControl->m_apiCode) << " errCode:" << errCodeToString(cameraControl->m_errCode) << std::endl;
}

void SelectErrorView()
{
	std::cout << std::endl;
	std::cout << "Please choose from the choices" << std::endl;
	
	if(std::cin.fail())
	{
		std::cin.clear();
		std::cin.ignore(std::numeric_limits<std::streamsize>::max(), '\n');
	}
}

bool CompareByIndex(const CameraInfo& info1, const CameraInfo& info2)
{
	return info1.index < info2.index;
}

