package com.fujifilm.zoompos

import com.fujifilm.zoompos.xsdk.XSDK
import com.fujifilm.zoompos.xsdk.XSDK.SDKLong
import com.fujifilm.zoompos.xsdk.XSDK.XSDK_ImageInformation

object SDKResult{
    const val XSDK_COMPLETE   = 0L
    const val XSDK_ERROR      = -1L
}

object Constants {
    // Image Format(ReadImageInfo)
    const val   XSDK_IMAGEFORMAT_RAW              = 1
    const val   XSDK_IMAGEFORMAT_LIVE             = 4
    const val   XSDK_IMAGEFORMAT_NONE             = 5
    const val   XSDK_IMAGEFORMAT_JPEG             = 7
    const val   XSDK_IMAGEFORMAT_TIFF             = 9
    const val   XSDK_IMAGEFORMAT_HEIF             = 0x0012
    const val	XSDK_IMAGEFORMAT_JPEG_90          = 0x0607
    const val	XSDK_IMAGEFORMAT_JPEG_180         = 0x0307
    const val	XSDK_IMAGEFORMAT_JPEG_270         = 0x0807
    const val	XSDK_IMAGEFORMAT_RAW_90           = 0x0601
    const val	XSDK_IMAGEFORMAT_RAW_180          = 0x0301
    const val	XSDK_IMAGEFORMAT_RAW_270          = 0x0801
    const val	XSDK_IMAGEFORMAT_LIVE_90          = 0x0604
    const val	XSDK_IMAGEFORMAT_LIVE_180         = 0x0304
    const val	XSDK_IMAGEFORMAT_LIVE_270         = 0x0804
    const val	XSDK_IMAGEFORMAT_TIFF_90          = 0x0609
    const val	XSDK_IMAGEFORMAT_TIFF_180         = 0x0309
    const val	XSDK_IMAGEFORMAT_TIFF_270         = 0x0809
    const val   XSDK_IMAGEFORMAT_HEIF_90          = 0x0612
    const val   XSDK_IMAGEFORMAT_HEIF_180         = 0x0312
    const val   XSDK_IMAGEFORMAT_HEIF_270         = 0x0812

    // DRIVE MODE
    const val   XSDK_DRIVE_MODE_INVALID             = 0xFFFF
    const val   XSDK_DRIVE_MODE_CH                  = 0x0002
    const val   XSDK_DRIVE_MODE_CL                  = 0x0003
    const val   XSDK_DRIVE_MODE_S                   = 0x0004
    const val   XSDK_DRIVE_MODE_MULTI_EXPOSURE      = 0x0005
    const val   XSDK_DRIVE_MODE_ADVFILTER           = 0x0006
    const val   XSDK_DRIVE_MODE_PANORAMA            = 0x0007
    const val   XSDK_DRIVE_MODE_MOVIE               = 0x0008
    const val   XSDK_DRIVE_MODE_HDR                 = 0x0009
    const val   XSDK_DRIVE_MODE_BKT_AE              = 0x000A
    const val   XSDK_DRIVE_MODE_BKT_ISO             = 0x000B
    const val   XSDK_DRIVE_MODE_BKT_FILMSIMULATION  = 0x000C
    const val   XSDK_DRIVE_MODE_BKT_WHITEBALANCE    = 0x000D
    const val   XSDK_DRIVE_MODE_BKT_DYNAMICRANGE    = 0x000E
    const val   XSDK_DRIVE_MODE_BKT_FOCUS           = 0x000F
    const val   XSDK_DRIVE_MODE_PIXELSHIFTMULTISHOT = 0x0010
    const val   XSDK_DRIVE_MODE_CH_CROP             = 0x0011
    const val   XSDK_DRIVE_MODE_PIXELSHIFTMULTISHOT_FEWERFRAMES = 0x0012

    // Release Mode Ex
    const val   XSDK_RELEASE_EX_S1_ON_S2_ON_S2_OFF_S1_OFF:Long	= 0x00030003
}

object CameraControl {

    private val xsdk: XSDK = XSDK()
    data class ExposureBias(val str: String, val num: Long)
    val ExposureList = listOf(
        ExposureBias("+5",150),
        ExposureBias("+4 2/3",140),
        ExposureBias("+4 1/3",130),
        ExposureBias("+4",120),
        ExposureBias("+3 2/3",110),
        ExposureBias("+3 1/3",100),
        ExposureBias("+3",90),
        ExposureBias("+2 2/3",80),
        ExposureBias("+2 1/3",70),
        ExposureBias("+2",60),
        ExposureBias("+1 2/3",50),
        ExposureBias("+1 1/3",40),
        ExposureBias("+1",30),
        ExposureBias("+2/3",20),
        ExposureBias("+1/3",10),
        ExposureBias("±0",0),
        ExposureBias("-1/3",-10),
        ExposureBias("-2/3",-20),
        ExposureBias("-1",-30),
        ExposureBias("-1 1/3",-40),
        ExposureBias("-1 2/3",-50),
        ExposureBias("-2",-60),
        ExposureBias("-2 1/3",-70),
        ExposureBias("-2 2/3",-80),
        ExposureBias("-3",-90),
        ExposureBias("-3 1/3",-100),
        ExposureBias("-3 2/3",-110),
        ExposureBias("-4",-120),
        ExposureBias("-4 1/3",-130),
        ExposureBias("-4 2/3",-140),
        ExposureBias("-5",-150),
    )

    data class APICode(val apiName: String, val apiCode: Long)
    val apiList = listOf(
        APICode("API_CODE_Init",0x1001),
        APICode("API_CODE_Exit",0x1002),
        APICode("API_CODE_Close",0x1022),
        APICode("API_CODE_GetErrorNumber",0x1031),
        APICode("API_CODE_GetDeviceInfo",0x1041),
        APICode("API_CODE_SetPriorityMode",0x1102),
        APICode("API_CODE_ReleaseEx",0x1116),
        APICode("API_CODE_ReadImageInfo",0x1201),
        APICode("API_CODE_ReadImage",0x1203),
        APICode("API_CODE_DeleteImage",0x1204),
        APICode("API_CODE_CapExposureBias",0x1307),
        APICode("API_CODE_SetExposureBias",0x1308),
        APICode("API_CODE_GetExposureBias",0x1309),
        APICode("API_CODE_CapLensZoomPos",0x1321),
        APICode("API_CODE_GetLensZoomPos",0x1323),
        APICode("API_CODE_GetDriveMode",0x1378),
        APICode("API_CODE_CapProp",0x1401),
        APICode("API_CODE_SetProp",0x1402),
        APICode("API_CODE_GetProp",0x1403),
        APICode("API_CODE_SetUSBDeviceHandle",0x1602),
        APICode("API_CODE_StartLiveView",0x3301),
        APICode("API_CODE_StopLiveView",0x3302),
        APICode("API_CODE_SetLiveViewImageQuality",0x3323),
        APICode("API_CODE_SetLiveViewImageSize",0x3325),
        APICode("API_CODE_SetMediaRecord",0x4066),
        )

    data class ErrorCode(val codeName: String, val errorCode: Long)
    private val errorList = listOf(
        ErrorCode("XSDK_ERRCODE_NOERR",0x00000000),
        ErrorCode("XSDK_ERRCODE_SEQUENCE",0x00001001),
        ErrorCode("XSDK_ERRCODE_PARAM",0x00001002),
        ErrorCode("XSDK_ERRCODE_INVALID_CAMERA",0x00001003),
        ErrorCode("XSDK_ERRCODE_LOADLIB",0x00001004),
        ErrorCode("XSDK_ERRCODE_UNSUPPORTED",0x00001005),
        ErrorCode("XSDK_ERRCODE_BUSY",0x00001006),
        ErrorCode("XSDK_ERRCODE_AF_TIMEOUT",0x00001007),
        ErrorCode("XSDK_ERRCODE_SHOOT_ERROR",0x00001008),
        ErrorCode("XSDK_ERRCODE_FRAME_FULL",0x00001009),
        ErrorCode("XSDK_ERRCODE_STANDBY",0x00001010),
        ErrorCode("XSDK_ERRCODE_NODRIVER",0x00001011),
        ErrorCode("XSDK_ERRCODE_NO_MODEL_MODULE",0x00001012),
        ErrorCode("XSDK_ERRCODE_API_NOTFOUND",0x00001013),
        ErrorCode("XSDK_ERRCODE_API_MISMATCH",0x00001014),
        ErrorCode("XSDK_ERRCODE_INVALID_USBMODE",0x00001015),
        ErrorCode("XSDK_ERRCODE_FORCEMODE_BUSY",0x00001016),
        ErrorCode("XSDK_ERRCODE_RUNNING_OTHER_FUNCTION",0x00001017),
        ErrorCode("XSDK_ERRCODE_COMMUNICATION",0x00002001),
        ErrorCode("XSDK_ERRCODE_TIMEOUT",0x00002002),
        ErrorCode("XSDK_ERRCODE_COMBINATION",0x00002003),
        ErrorCode("XSDK_ERRCODE_WRITEERROR",0x00002004),
        ErrorCode("XSDK_ERRCODE_CARDFULL",0x00002005),
        ErrorCode("XSDK_ERRCODE_HARDWARE",0x00003001),
        ErrorCode("XSDK_ERRCODE_INTERNAL",0x00009001),
        ErrorCode("XSDK_ERRCODE_MEMFULL",0x00009002),
        ErrorCode("XSDK_ERRCODE_UNKNOWN",0x00009100),
    )

    @Synchronized
    fun init():Long {
        if (xsdk.XSDK_loadLibrary() != SDKResult.XSDK_COMPLETE) return SDKResult.XSDK_ERROR
        return xsdk.XSDK_Init()
    }

    @Synchronized
    fun exit():Long {
        return xsdk.XSDK_Exit()
    }

    @Synchronized
    fun setUSBDeviceHandle(fileDescriptor:Long, descriptors:ByteArray, hCamera: XSDK.SDKLong):Long {
        return xsdk.XSDK_SetUSBDeviceHandle(fileDescriptor, descriptors, hCamera)
    }

    @Synchronized
    fun close(hCamera:Long):Long {
        return xsdk.XSDK_Close(hCamera)
    }

    @Synchronized
    fun getError(hCamera:Long):String {
        val apiCode = XSDK.SDKLong(0)
        val errorCode = XSDK.SDKLong(0)
        val ret = xsdk.XSDK_GetErrorNumber(hCamera, apiCode, errorCode)
        return if (ret == SDKResult.XSDK_COMPLETE) {
            val apiName = apiList.find { it.apiCode == apiCode.long }
            val errorName = errorList.find { it.errorCode == errorCode.long }
            val apiStr = apiName?.apiName ?: ("0x" + apiCode.long.toString(16))
            val errorStr = errorName?.codeName ?: ("0x" + errorCode.long.toString(16))

            "APICode: $apiStr\nErrorCode: $errorStr"
        }else{
            "Error"
        }
    }

    @Synchronized
    fun getDeviceInfo(hCamera:Long, pDevInfo: XSDK.XSDK_DeviceInformation):Long {
        return xsdk.XSDK_GetDeviceInfo(hCamera, pDevInfo)
    }

    @Synchronized
    fun release(hCamera:Long, ulReleaseMode:Long, pShotOpt: SDKLong, pStatus: SDKLong):Long {
        return xsdk.XSDK_ReleaseEx(hCamera, ulReleaseMode, pShotOpt, pStatus)
    }

    @Synchronized
    fun readImageInfo(hCamera:Long, pImgInfoArray: XSDK_ImageInformation):Long {
        return xsdk.XSDK_ReadImageInfo(hCamera, pImgInfoArray)
    }

    @Synchronized
    fun readImage(hCamera:Long, pData:ByteArray, dataSize:Long):Long {
        return xsdk.XSDK_ReadImage(hCamera, pData, dataSize)
    }

    @Synchronized
    fun deleteImage(hCamera:Long):Long {
        return xsdk.XSDK_DeleteImage(hCamera)
    }

    @Synchronized
    fun setPriorityMode(hCamera:Long, lPriorityMode:Long):Long {
        return xsdk.XSDK_SetPriorityMode(hCamera, lPriorityMode)
    }

    @Synchronized
    fun capExposureBias(hCamera:Long, plNumExposureBias: SDKLong, plExposureBias: MutableList<Long>):Long {
        var ret = xsdk.XSDK_CapExposureBias(hCamera, plNumExposureBias, null)
        if(ret != SDKResult.XSDK_COMPLETE) return ret

        val values = LongArray(plNumExposureBias.long.toInt())
        ret = xsdk.XSDK_CapExposureBias(hCamera, plNumExposureBias, values)
        if(ret != SDKResult.XSDK_COMPLETE) return ret
        for(i in 0 until plNumExposureBias.long.toInt()){
            plExposureBias.add(values[i])
        }
        return SDKResult.XSDK_COMPLETE
    }

    @Synchronized
    fun setExposureBias(hCamera:Long, lExposureBias:Long):Long {
        return xsdk.XSDK_SetExposureBias(hCamera, lExposureBias)
    }

    @Synchronized
    fun getExposureBias(hCamera:Long, plExposureBias: SDKLong):Long {
        return xsdk.XSDK_GetExposureBias(hCamera, plExposureBias)
    }

    @Synchronized
    fun capLensZoomPos(hCamera:Long, plNumZoomPos: SDKLong, plZoomPos:MutableList<Long>, plFocusLength:MutableList<Long>, pl35mmFocusLength:MutableList<Long>):Long {
        var ret = xsdk.XSDK_CapLensZoomPos(hCamera, plNumZoomPos, null, null,null)
        if(ret != SDKResult.XSDK_COMPLETE) return ret

        val zoomPosArray = LongArray(plNumZoomPos.long.toInt())
        val focusLengthArray = LongArray(plNumZoomPos.long.toInt())
        val focusLength35mmArray = LongArray(plNumZoomPos.long.toInt())

        ret = xsdk.XSDK_CapLensZoomPos(hCamera, plNumZoomPos, zoomPosArray, focusLengthArray, focusLength35mmArray)
        if(ret != SDKResult.XSDK_COMPLETE) return ret
        for(i in 0 until plNumZoomPos.long.toInt()){
            plZoomPos.add(zoomPosArray[i])
            plFocusLength.add(focusLengthArray[i])
            pl35mmFocusLength.add(focusLength35mmArray[i])
        }
        return SDKResult.XSDK_COMPLETE
    }

    @Synchronized
    fun getLensZoomPos(hCamera:Long, plZoomPos: SDKLong):Long {
        return xsdk.XSDK_GetLensZoomPos(hCamera, plZoomPos)
    }

    @Synchronized
    fun getDriveMode(hCamera:Long, plDriveMode: SDKLong):Long {
        var ret = xsdk.XSDK_GetDriveMode(hCamera, plDriveMode)
        if(ret != SDKResult.XSDK_COMPLETE) return ret
        if(plDriveMode.long == Constants.XSDK_DRIVE_MODE_MOVIE.toLong())
        {
            return ret
        }
        val plMode = SDKLong(0)
        ret = xsdk.XSDK_GetMode(hCamera, plMode)
        if(ret != SDKResult.XSDK_COMPLETE) return ret
        if ((plMode.long and 0x0100L) == 0x0100L) {
            plDriveMode.long = Constants.XSDK_DRIVE_MODE_MOVIE.toLong()
        }
        return ret
    }

    @Synchronized
    fun setProp(hCamera:Long, lAPICode:Long, lAPIParam:Long):Long {
        return xsdk.XSDK_SetProp(hCamera, lAPICode, lAPIParam)
    }

    @Synchronized
    fun setProp_L(hCamera:Long, lAPICode:Long, lAPIParam:Long, param1:Long):Long {
        return xsdk.XSDK_SetProp_L(hCamera, lAPICode, lAPIParam, param1)
    }

    @Synchronized
    fun setProp_LL(hCamera:Long, lAPICode:Long, lAPIParam:Long, param1:Long, param2:Long):Long {
        return  xsdk.XSDK_SetProp_LL(hCamera, lAPICode, lAPIParam, param1, param2)
    }

    @Synchronized
    fun getProp_PP(hCamera:Long, lAPICode:Long, lAPIParam:Long, param1:SDKLong, param2:SDKLong):Long {
        return  xsdk.XSDK_GetProp_PP(hCamera, lAPICode, lAPIParam, param1, param2)
    }
}