//
//  FujiSDK.mm
//  FujiSDK
//
//  Created by DVN405 on 2024/10/07.
//

#import <Foundation/Foundation.h>
#import <CoreFoundation/CoreFoundation.h>
#import "FujiSDK.h"

@implementation CameraList
@synthesize strProduct;
@synthesize strSerialNo;
@synthesize strIPAddress;
@synthesize strFramework;
@synthesize bValid;
@end

@implementation DeviceInformation
@synthesize strVendor;
@synthesize strManufacturer;
@synthesize strProduct;
@synthesize strFirmware;
@synthesize strDeviceType;
@synthesize strSerialNo;
@synthesize strFramework;
@synthesize bDeviceId;
@synthesize strDeviceName;
@synthesize strYNo;
@end

@implementation LensInformation
@synthesize strModel;
@synthesize strProductName;
@synthesize strSerialNo;
@synthesize lISCapability;
@synthesize lMFCapability;
@synthesize lZoomPosCapability;
@end

@implementation ImageInformation
@synthesize strInternalName;
@synthesize lFormat;
@synthesize lDataSize;
@synthesize lImagePixHeight;
@synthesize lImagePixWidth;
@synthesize lImageBitDepth;
@synthesize lPreviewSize;
@synthesize hImage;
@end


@implementation FujiSDK
{
    LIB_HANDLE bundleHandle;
    _XSDK_Init                 pXSDK_Init;
    _XSDK_Exit                 pXSDK_Exit;
    _XSDK_Detect               pXSDK_Detect;
    _XSDK_Append               pXSDK_Append;
    _XSDK_Close                pXSDK_Close;
    _XSDK_PowerOFF             pXSDK_PowerOFF;
    _XSDK_OpenEx               pXSDK_OpenEx;
    _XSDK_GetErrorNumber       pXSDK_GetErrorNumber;
    _XSDK_GetVersionString     pXSDK_GetVersionString;
    _XSDK_GetErrorDetails      pXSDK_GetErrorDetails;
    _XSDK_GetDeviceInfo        pXSDK_GetDeviceInfo;
    _XSDK_WriteDeviceName      pXSDK_WriteDeviceName;
    _XSDK_GetFirmwareVersion   pXSDK_GetFirmwareVersion;
    _XSDK_GetLensInfo          pXSDK_GetLensInfo;
    _XSDK_GetLensVersion       pXSDK_GetLensVersion;
    _XSDK_GetDeviceInfoEx      pXSDK_GetDeviceInfoEx;
    _XSDK_CapPriorityMode      pXSDK_CapPriorityMode;
    _XSDK_SetPriorityMode      pXSDK_SetPriorityMode;
    _XSDK_GetPriorityMode      pXSDK_GetPriorityMode;
    _XSDK_CapRelease           pXSDK_CapRelease;
    _XSDK_Release              pXSDK_Release;
    _XSDK_GetReleaseStatus     pXSDK_GetReleaseStatus;
    _XSDK_CapReleaseEx         pXSDK_CapReleaseEx;
    _XSDK_ReleaseEx            pXSDK_ReleaseEx;
    _XSDK_GetBufferCapacity    pXSDK_GetBufferCapacity;
    _XSDK_ReadImageInfo        pXSDK_ReadImageInfo;
    _XSDK_ReadPreview          pXSDK_ReadPreview;
    _XSDK_ReadImage            pXSDK_ReadImage;
    _XSDK_DeleteImage          pXSDK_DeleteImage;
    _XSDK_CapAEMode            pXSDK_CapAEMode;
    _XSDK_SetAEMode            pXSDK_SetAEMode;
    _XSDK_GetAEMode            pXSDK_GetAEMode;
    _XSDK_CapShutterSpeed      pXSDK_CapShutterSpeed;
    _XSDK_SetShutterSpeed      pXSDK_SetShutterSpeed;
    _XSDK_GetShutterSpeed      pXSDK_GetShutterSpeed;
    _XSDK_CapExposureBias      pXSDK_CapExposureBias;
    _XSDK_SetExposureBias      pXSDK_SetExposureBias;
    _XSDK_GetExposureBias      pXSDK_GetExposureBias;
    _XSDK_CapSensitivity       pXSDK_CapSensitivity;
    _XSDK_SetSensitivity       pXSDK_SetSensitivity;
    _XSDK_GetSensitivity       pXSDK_GetSensitivity;
    _XSDK_CapDynamicRange      pXSDK_CapDynamicRange;
    _XSDK_SetDynamicRange      pXSDK_SetDynamicRange;
    _XSDK_GetDynamicRange      pXSDK_GetDynamicRange;
    _XSDK_CapMeteringMode      pXSDK_CapMeteringMode;
    _XSDK_SetMeteringMode      pXSDK_SetMeteringMode;
    _XSDK_GetMeteringMode      pXSDK_GetMeteringMode;
    _XSDK_CapLensZoomPos       pXSDK_CapLensZoomPos;
    _XSDK_SetLensZoomPos       pXSDK_SetLensZoomPos;
    _XSDK_GetLensZoomPos       pXSDK_GetLensZoomPos;
    _XSDK_CapAperture          pXSDK_CapAperture;
    _XSDK_SetAperture          pXSDK_SetAperture;
    _XSDK_GetAperture          pXSDK_GetAperture;
    _XSDK_CapWBMode            pXSDK_CapWBMode;
    _XSDK_SetWBMode            pXSDK_SetWBMode;
    _XSDK_GetWBMode            pXSDK_GetWBMode;
    _XSDK_CapWBColorTemp       pXSDK_CapWBColorTemp;
    _XSDK_SetWBColorTemp       pXSDK_SetWBColorTemp;
    _XSDK_GetWBColorTemp       pXSDK_GetWBColorTemp;
    _XSDK_CapMediaRecord       pXSDK_CapMediaRecord;
    _XSDK_SetMediaRecord       pXSDK_SetMediaRecord;
    _XSDK_GetMediaRecord       pXSDK_GetMediaRecord;
    _XSDK_CapForceMode         pXSDK_CapForceMode;
    _XSDK_SetForceMode         pXSDK_SetForceMode;
    _XSDK_SetBackupSettings    pXSDK_SetBackupSettings;
    _XSDK_GetBackupSettings    pXSDK_GetBackupSettings;
    _XSDK_SetDriveMode         pXSDK_SetDriveMode;
    _XSDK_GetDriveMode         pXSDK_GetDriveMode;
    _XSDK_CapDriveMode         pXSDK_CapDriveMode;
    _XSDK_CapMode              pXSDK_CapMode;
    _XSDK_SetMode              pXSDK_SetMode;
    _XSDK_GetMode              pXSDK_GetMode;
    _XSDK_GetRecordingStatus   pXSDK_GetRecordingStatus;
    _XSDK_CapMovieShutterSpeed pXSDK_CapMovieShutterSpeed;
    _XSDK_SetMovieShutterSpeed pXSDK_SetMovieShutterSpeed;
    _XSDK_GetMovieShutterSpeed pXSDK_GetMovieShutterSpeed;
    _XSDK_CapMovieExposureBias pXSDK_CapMovieExposureBias;
    _XSDK_SetMovieExposureBias pXSDK_SetMovieExposureBias;
    _XSDK_GetMovieExposureBias pXSDK_GetMovieExposureBias;
    _XSDK_CapMovieSensitivity  pXSDK_CapMovieSensitivity;
    _XSDK_SetMovieSensitivity  pXSDK_SetMovieSensitivity;
    _XSDK_GetMovieSensitivity  pXSDK_GetMovieSensitivity;
    _XSDK_CapMovieAperture     pXSDK_CapMovieAperture;
    _XSDK_SetMovieAperture     pXSDK_SetMovieAperture;
    _XSDK_GetMovieAperture     pXSDK_GetMovieAperture;
    _XSDK_CapMovieDynamicRange pXSDK_CapMovieDynamicRange;
    _XSDK_SetMovieDynamicRange pXSDK_SetMovieDynamicRange;
    _XSDK_GetMovieDynamicRange pXSDK_GetMovieDynamicRange;
    _XSDK_CapMovieMeteringMode pXSDK_CapMovieMeteringMode;
    _XSDK_SetMovieMeteringMode pXSDK_SetMovieMeteringMode;
    _XSDK_GetMovieMeteringMode pXSDK_GetMovieMeteringMode;
    _XSDK_CapMovieWBMode       pXSDK_CapMovieWBMode;
    _XSDK_SetMovieWBMode       pXSDK_SetMovieWBMode;
    _XSDK_GetMovieWBMode       pXSDK_GetMovieWBMode;
    _XSDK_CapMovieWBColorTemp  pXSDK_CapMovieWBColorTemp;
    _XSDK_SetMovieWBColorTemp  pXSDK_SetMovieWBColorTemp;
    _XSDK_GetMovieWBColorTemp  pXSDK_GetMovieWBColorTemp;
    _XSDK_CapProp              pXSDK_CapProp;
    _XSDK_SetProp              pXSDK_SetProp;
    _XSDK_GetProp              pXSDK_GetProp;
}

-(instancetype)init
{
    self = [super init];
    if (self)
    {
        bool succeeded = false;
        
        CFBundleRef mainBundle = ::CFBundleGetMainBundle();
        if (mainBundle)
        {
            CFStringRef libName = ::CFStringCreateWithCString(kCFAllocatorDefault, "XAPI.bundle", ::CFStringGetSystemEncoding());
            if (libName)
            {
                CFURLRef libURL = ::CFBundleCopyResourceURL(mainBundle, libName, NULL, NULL);
                if (libURL)
                {
                    bundleHandle = ::CFBundleCreate(kCFAllocatorDefault, libURL);
                    if (bundleHandle)
                    {
                        succeeded = ::CFBundleLoadExecutable(bundleHandle);
                    }
                    ::CFRelease(libURL);
                }
                ::CFRelease(libName);
            }
        }
        
        if(!succeeded)
        {
            // error
        }
    }
    return self;
}

-(void)dealloc
{
    if(bundleHandle)
    {
        if (::CFBundleIsExecutableLoaded(bundleHandle))
        {
            ::CFBundleUnloadExecutable(bundleHandle);
        }
        ::CFRelease(bundleHandle);
    }
}

/**
 * @brief Load library
 *
 * @return Load library result.
 */
-(bool)LoadLibrary
{
    if(bundleHandle)
    {
        try
        {
            pXSDK_Init = (_XSDK_Init)[self GetProcAddress:bundleHandle pProcName:"XSDK_Init"];
            if(!pXSDK_Init) throw;
            pXSDK_Exit = (_XSDK_Exit)[self GetProcAddress:bundleHandle pProcName:"XSDK_Exit"];
            if(! pXSDK_Exit) throw;
            pXSDK_Detect = (_XSDK_Detect)[self GetProcAddress:bundleHandle pProcName:"XSDK_Detect"];
            if(! pXSDK_Detect) throw;
            pXSDK_Append = (_XSDK_Append)[self GetProcAddress:bundleHandle pProcName:"XSDK_Append"];
            if(! pXSDK_Append) throw;
            pXSDK_Close = (_XSDK_Close)[self GetProcAddress:bundleHandle pProcName:"XSDK_Close"];
            if(! pXSDK_Close) throw;
            pXSDK_PowerOFF = (_XSDK_PowerOFF)[self GetProcAddress:bundleHandle pProcName:"XSDK_PowerOFF"];
            if(! pXSDK_PowerOFF) throw;
            pXSDK_OpenEx = (_XSDK_OpenEx)[self GetProcAddress:bundleHandle pProcName:"XSDK_OpenEx"];
            if(! pXSDK_OpenEx) throw;
            pXSDK_GetErrorNumber = (_XSDK_GetErrorNumber)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetErrorNumber"];
            if(! pXSDK_GetErrorNumber) throw;
            pXSDK_GetVersionString = (_XSDK_GetVersionString)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetVersionString"];
            if(! pXSDK_GetVersionString) throw;
            pXSDK_GetErrorDetails = (_XSDK_GetErrorDetails)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetErrorDetails"];
            if(! pXSDK_GetErrorDetails) throw;
            pXSDK_GetDeviceInfo = (_XSDK_GetDeviceInfo)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetDeviceInfo"];
            if(! pXSDK_GetDeviceInfo) throw;
            pXSDK_WriteDeviceName = (_XSDK_WriteDeviceName)[self GetProcAddress:bundleHandle pProcName:"XSDK_WriteDeviceName"];
            if(! pXSDK_WriteDeviceName) throw;
            pXSDK_GetFirmwareVersion = (_XSDK_GetFirmwareVersion)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetFirmwareVersion"];
            if(! pXSDK_GetFirmwareVersion) throw;
            pXSDK_GetLensInfo = (_XSDK_GetLensInfo)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetLensInfo"];
            if(! pXSDK_GetLensInfo) throw;
            pXSDK_GetLensVersion = (_XSDK_GetLensVersion)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetLensVersion"];
            if(! pXSDK_GetLensVersion) throw;
            pXSDK_GetDeviceInfoEx = (_XSDK_GetDeviceInfoEx)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetDeviceInfoEx"];
            if(! pXSDK_GetDeviceInfoEx) throw;
            pXSDK_CapPriorityMode = (_XSDK_CapPriorityMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapPriorityMode"];
            if(! pXSDK_CapPriorityMode) throw;
            pXSDK_SetPriorityMode = (_XSDK_SetPriorityMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetPriorityMode"];
            if(! pXSDK_SetPriorityMode) throw;
            pXSDK_GetPriorityMode = (_XSDK_GetPriorityMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetPriorityMode"];
            if(! pXSDK_GetPriorityMode) throw;
            pXSDK_CapRelease = (_XSDK_CapRelease)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapRelease"];
            if(! pXSDK_CapRelease) throw;
            pXSDK_Release = (_XSDK_Release)[self GetProcAddress:bundleHandle pProcName:"XSDK_Release"];
            if(! pXSDK_Release) throw;
            pXSDK_GetReleaseStatus = (_XSDK_GetReleaseStatus)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetReleaseStatus"];
            if(! pXSDK_GetReleaseStatus) throw;
            pXSDK_CapReleaseEx = (_XSDK_CapReleaseEx)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapReleaseEx"];
            if(! pXSDK_CapReleaseEx) throw;
            pXSDK_ReleaseEx = (_XSDK_ReleaseEx)[self GetProcAddress:bundleHandle pProcName:"XSDK_ReleaseEx"];
            if(! pXSDK_ReleaseEx) throw;
            pXSDK_GetBufferCapacity = (_XSDK_GetBufferCapacity)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetBufferCapacity"];
            if(! pXSDK_GetBufferCapacity) throw;
            pXSDK_ReadImageInfo = (_XSDK_ReadImageInfo)[self GetProcAddress:bundleHandle pProcName:"XSDK_ReadImageInfo"];
            if(! pXSDK_ReadImageInfo) throw;
            pXSDK_ReadPreview = (_XSDK_ReadPreview)[self GetProcAddress:bundleHandle pProcName:"XSDK_ReadPreview"];
            if(! pXSDK_ReadPreview) throw;
            pXSDK_ReadImage = (_XSDK_ReadImage)[self GetProcAddress:bundleHandle pProcName:"XSDK_ReadImage"];
            if(! pXSDK_ReadImage) throw;
            pXSDK_DeleteImage = (_XSDK_DeleteImage)[self GetProcAddress:bundleHandle pProcName:"XSDK_DeleteImage"];
            if(! pXSDK_DeleteImage) throw;
            pXSDK_CapAEMode = (_XSDK_CapAEMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapAEMode"];
            if(! pXSDK_CapAEMode) throw;
            pXSDK_SetAEMode = (_XSDK_SetAEMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetAEMode"];
            if(! pXSDK_SetAEMode) throw;
            pXSDK_GetAEMode = (_XSDK_GetAEMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetAEMode"];
            if(! pXSDK_GetAEMode) throw;
            pXSDK_CapShutterSpeed = (_XSDK_CapShutterSpeed)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapShutterSpeed"];
            if(! pXSDK_CapShutterSpeed) throw;
            pXSDK_SetShutterSpeed = (_XSDK_SetShutterSpeed)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetShutterSpeed"];
            if(! pXSDK_SetShutterSpeed) throw;
            pXSDK_GetShutterSpeed = (_XSDK_GetShutterSpeed)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetShutterSpeed"];
            if(! pXSDK_GetShutterSpeed) throw;
            pXSDK_CapExposureBias = (_XSDK_CapExposureBias)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapExposureBias"];
            if(! pXSDK_CapExposureBias) throw;
            pXSDK_SetExposureBias = (_XSDK_SetExposureBias)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetExposureBias"];
            if(! pXSDK_SetExposureBias) throw;
            pXSDK_GetExposureBias = (_XSDK_GetExposureBias)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetExposureBias"];
            if(! pXSDK_GetExposureBias) throw;
            pXSDK_CapSensitivity = (_XSDK_CapSensitivity)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapSensitivity"];
            if(! pXSDK_CapSensitivity) throw;
            pXSDK_SetSensitivity = (_XSDK_SetSensitivity)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetSensitivity"];
            if(! pXSDK_SetSensitivity) throw;
            pXSDK_GetSensitivity = (_XSDK_GetSensitivity)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetSensitivity"];
            if(! pXSDK_GetSensitivity) throw;
            pXSDK_CapDynamicRange = (_XSDK_CapDynamicRange)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapDynamicRange"];
            if(! pXSDK_CapDynamicRange) throw;
            pXSDK_SetDynamicRange = (_XSDK_SetDynamicRange)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetDynamicRange"];
            if(! pXSDK_SetDynamicRange) throw;
            pXSDK_GetDynamicRange = (_XSDK_GetDynamicRange)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetDynamicRange"];
            if(! pXSDK_GetDynamicRange) throw;
            pXSDK_CapMeteringMode = (_XSDK_CapMeteringMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapMeteringMode"];
            if(! pXSDK_CapMeteringMode) throw;
            pXSDK_SetMeteringMode = (_XSDK_SetMeteringMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetMeteringMode"];
            if(! pXSDK_SetMeteringMode) throw;
            pXSDK_GetMeteringMode = (_XSDK_GetMeteringMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetMeteringMode"];
            if(! pXSDK_GetMeteringMode) throw;
            pXSDK_CapLensZoomPos = (_XSDK_CapLensZoomPos)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapLensZoomPos"];
            if(! pXSDK_CapLensZoomPos) throw;
            pXSDK_SetLensZoomPos = (_XSDK_SetLensZoomPos)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetLensZoomPos"];
            if(! pXSDK_SetLensZoomPos) throw;
            pXSDK_GetLensZoomPos = (_XSDK_GetLensZoomPos)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetLensZoomPos"];
            if(! pXSDK_GetLensZoomPos) throw;
            pXSDK_CapAperture = (_XSDK_CapAperture)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapAperture"];
            if(! pXSDK_CapAperture) throw;
            pXSDK_SetAperture = (_XSDK_SetAperture)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetAperture"];
            if(! pXSDK_SetAperture) throw;
            pXSDK_GetAperture = (_XSDK_GetAperture)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetAperture"];
            if(! pXSDK_GetAperture) throw;
            pXSDK_CapWBMode = (_XSDK_CapWBMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapWBMode"];
            if(! pXSDK_CapWBMode) throw;
            pXSDK_SetWBMode = (_XSDK_SetWBMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetWBMode"];
            if(! pXSDK_SetWBMode) throw;
            pXSDK_GetWBMode = (_XSDK_GetWBMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetWBMode"];
            if(! pXSDK_GetWBMode) throw;
            pXSDK_CapWBColorTemp = (_XSDK_CapWBColorTemp)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapWBColorTemp"];
            if(! pXSDK_CapWBColorTemp) throw;
            pXSDK_SetWBColorTemp = (_XSDK_SetWBColorTemp)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetWBColorTemp"];
            if(! pXSDK_SetWBColorTemp) throw;
            pXSDK_GetWBColorTemp = (_XSDK_GetWBColorTemp)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetWBColorTemp"];
            if(! pXSDK_GetWBColorTemp) throw;
            pXSDK_CapMediaRecord = (_XSDK_CapMediaRecord)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapMediaRecord"];
            if(! pXSDK_CapMediaRecord) throw;
            pXSDK_SetMediaRecord = (_XSDK_SetMediaRecord)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetMediaRecord"];
            if(! pXSDK_SetMediaRecord) throw;
            pXSDK_GetMediaRecord = (_XSDK_GetMediaRecord)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetMediaRecord"];
            if(! pXSDK_GetMediaRecord) throw;
            pXSDK_CapForceMode = (_XSDK_CapForceMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapForceMode"];
            if(! pXSDK_CapForceMode) throw;
            pXSDK_SetForceMode = (_XSDK_SetForceMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetForceMode"];
            if(! pXSDK_SetForceMode) throw;
            pXSDK_SetBackupSettings = (_XSDK_SetBackupSettings)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetBackupSettings"];
            if(! pXSDK_SetBackupSettings) throw;
            pXSDK_GetBackupSettings = (_XSDK_GetBackupSettings)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetBackupSettings"];
            if(! pXSDK_GetBackupSettings) throw;
            pXSDK_SetDriveMode = (_XSDK_SetDriveMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetDriveMode"];
            if(! pXSDK_SetDriveMode) throw;
            pXSDK_GetDriveMode = (_XSDK_GetDriveMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetDriveMode"];
            if(! pXSDK_GetDriveMode) throw;
            pXSDK_CapDriveMode = (_XSDK_CapDriveMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapDriveMode"];
            if(! pXSDK_CapDriveMode) throw;
            pXSDK_CapMode = (_XSDK_CapMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapMode"];
            if(! pXSDK_CapMode) throw;
            pXSDK_SetMode = (_XSDK_SetMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetMode"];
            if(! pXSDK_SetMode) throw;
            pXSDK_GetMode = (_XSDK_GetMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetMode"];
            if(! pXSDK_GetMode) throw;
            pXSDK_GetRecordingStatus = (_XSDK_GetRecordingStatus)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetRecordingStatus"];
            if(! pXSDK_GetRecordingStatus) throw;
            pXSDK_CapMovieShutterSpeed = (_XSDK_CapMovieShutterSpeed)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapMovieShutterSpeed"];
            if(! pXSDK_CapMovieShutterSpeed) throw;
            pXSDK_SetMovieShutterSpeed = (_XSDK_SetMovieShutterSpeed)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetMovieShutterSpeed"];
            if(! pXSDK_SetMovieShutterSpeed) throw;
            pXSDK_GetMovieShutterSpeed = (_XSDK_GetMovieShutterSpeed)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetMovieShutterSpeed"];
            if(! pXSDK_GetMovieShutterSpeed) throw;
            pXSDK_CapMovieExposureBias = (_XSDK_CapMovieExposureBias)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapMovieExposureBias"];
            if(! pXSDK_CapMovieExposureBias) throw;
            pXSDK_SetMovieExposureBias = (_XSDK_SetMovieExposureBias)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetMovieExposureBias"];
            if(! pXSDK_SetMovieExposureBias) throw;
            pXSDK_GetMovieExposureBias = (_XSDK_GetMovieExposureBias)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetMovieExposureBias"];
            if(! pXSDK_GetMovieExposureBias) throw;
            pXSDK_CapMovieSensitivity = (_XSDK_CapMovieSensitivity)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapMovieSensitivity"];
            if(! pXSDK_CapMovieSensitivity) throw;
            pXSDK_SetMovieSensitivity = (_XSDK_SetMovieSensitivity)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetMovieSensitivity"];
            if(! pXSDK_SetMovieSensitivity) throw;
            pXSDK_GetMovieSensitivity = (_XSDK_GetMovieSensitivity)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetMovieSensitivity"];
            if(! pXSDK_GetMovieSensitivity) throw;
            pXSDK_CapMovieAperture = (_XSDK_CapMovieAperture)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapMovieAperture"];
            if(! pXSDK_CapMovieAperture) throw;
            pXSDK_SetMovieAperture = (_XSDK_SetMovieAperture)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetMovieAperture"];
            if(! pXSDK_SetMovieAperture) throw;
            pXSDK_GetMovieAperture = (_XSDK_GetMovieAperture)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetMovieAperture"];
            if(! pXSDK_GetMovieAperture) throw;
            pXSDK_CapMovieDynamicRange = (_XSDK_CapMovieDynamicRange)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapMovieDynamicRange"];
            if(! pXSDK_CapMovieDynamicRange) throw;
            pXSDK_SetMovieDynamicRange = (_XSDK_SetMovieDynamicRange)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetMovieDynamicRange"];
            if(! pXSDK_SetMovieDynamicRange) throw;
            pXSDK_GetMovieDynamicRange = (_XSDK_GetMovieDynamicRange)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetMovieDynamicRange"];
            if(! pXSDK_GetMovieDynamicRange) throw;
            pXSDK_CapMovieMeteringMode = (_XSDK_CapMovieMeteringMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapMovieMeteringMode"];
            if(! pXSDK_CapMovieMeteringMode) throw;
            pXSDK_SetMovieMeteringMode = (_XSDK_SetMovieMeteringMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetMovieMeteringMode"];
            if(! pXSDK_SetMovieMeteringMode) throw;
            pXSDK_GetMovieMeteringMode = (_XSDK_GetMovieMeteringMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetMovieMeteringMode"];
            if(! pXSDK_GetMovieMeteringMode) throw;
            pXSDK_CapMovieWBMode = (_XSDK_CapMovieWBMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapMovieWBMode"];
            if(! pXSDK_CapMovieWBMode) throw;
            pXSDK_SetMovieWBMode = (_XSDK_SetMovieWBMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetMovieWBMode"];
            if(! pXSDK_SetMovieWBMode) throw;
            pXSDK_GetMovieWBMode = (_XSDK_GetMovieWBMode)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetMovieWBMode"];
            if(! pXSDK_GetMovieWBMode) throw;
            pXSDK_CapMovieWBColorTemp = (_XSDK_CapMovieWBColorTemp)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapMovieWBColorTemp"];
            if(! pXSDK_CapMovieWBColorTemp) throw;
            pXSDK_SetMovieWBColorTemp = (_XSDK_SetMovieWBColorTemp)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetMovieWBColorTemp"];
            if(! pXSDK_SetMovieWBColorTemp) throw;
            pXSDK_GetMovieWBColorTemp = (_XSDK_GetMovieWBColorTemp)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetMovieWBColorTemp"];
            if(! pXSDK_GetMovieWBColorTemp) throw;
            pXSDK_CapProp = (_XSDK_CapProp)[self GetProcAddress:bundleHandle pProcName:"XSDK_CapProp"];
            if(! pXSDK_CapProp) throw;
            pXSDK_SetProp = (_XSDK_SetProp)[self GetProcAddress:bundleHandle pProcName:"XSDK_SetProp"];
            if(! pXSDK_SetProp) throw;
            pXSDK_GetProp = (_XSDK_GetProp)[self GetProcAddress:bundleHandle pProcName:"XSDK_GetProp"];
            if(! pXSDK_GetProp) throw;
        }
        catch (...)
        {
            return false;
        }
    }
    else
    {
        return false;
    }
    return true;
}

-(void*)GetProcAddress:(LIB_HANDLE) hLib
             pProcName:(const char*) pProcName
{
    void*       result = NULL;
    CFBundleRef bundle = hLib;
    bool        loaded = false;
    
    if (bundle == NULL)
    {
        bundle = ::CFBundleGetMainBundle();
    }
    
    if (bundle)
    {
        loaded = ::CFBundleIsExecutableLoaded(bundle);
        if (!loaded)
        {
            loaded = ::CFBundleLoadExecutable(bundle);
        }
        if (loaded)
        {
            CFStringRef functionName = ::CFStringCreateWithCString(kCFAllocatorDefault, pProcName, ::CFStringGetSystemEncoding());
            if (functionName)
            {
                result = ::CFBundleGetFunctionPointerForName(bundle, functionName);
                ::CFRelease(functionName);
            }
        }
    }
    
    return result;
}

/**
 * @brief Initializes and starts use of the SDK.
 *
 * @return API result.
 */
-(long) XSDK_Init
{
    long result = pXSDK_Init(bundleHandle);
    
    return result;
}

/**
 * @brief Finalizes and terminates use of the SDK.
 *
 * @return API result.
 */
-(long) XSDK_Exit
{
    long result = pXSDK_Exit();
    
    return result;
}

/**
 * @brief Enumerates available cameras and generate a connected camera list.
 *
 * @param lInterface How to search for a camera.
 * @param pInterface Search criteria IP address.
 * @param pDeviceName Search criteria device name.
 * @param plCount Returns the number of devices detected.
 * @return API result.
 */
-(long) XSDK_Detect:(long) lInterface
         pInterface:(NSString*) pInterface
        pDeviceName:(NSString*) pDeviceName
            plCount:(long*) plCount
{
    char* strInterface = (char*)[pInterface UTF8String];
    char* strDeviceName = (char*)[pDeviceName UTF8String];
    
    long result = pXSDK_Detect(lInterface, strInterface, strDeviceName, plCount);
    if(result == 0)
    {
        pInterface = [NSString stringWithCString:strInterface encoding:NSUTF8StringEncoding];
        pDeviceName = [NSString stringWithCString:strDeviceName encoding:NSUTF8StringEncoding];
    }

    return result;
}

/**
 * @brief Update the connected camera list.
 *
 * @param lInterface How to search for a camera.
 * @param pInterface Search criteria IP address.
 * @param pDeviceName Search criteria device name.
 * @param plCount Returns the number of devices detected.
 * @param pCameraList Returns camera information.
 * @return API result.
 */
-(long) XSDK_Append:(long) lInterface
         pInterface:(NSString*) pInterface
        pDeviceName:(NSString*) pDeviceName
            plCount:(long*) plCount
        pCameraList:(NSArray<CameraList*>*) pCameraList
{
    char* strInterface = (char*)[pInterface UTF8String];
    char* strDeviceName = (char*)[pDeviceName UTF8String];
    XSDK_CameraList cameraList[*plCount];
    
    long result = pXSDK_Append(lInterface, strInterface, strDeviceName, plCount, cameraList);
    if(result == 0)
    {
        pInterface = [NSString stringWithCString:strInterface encoding:NSUTF8StringEncoding];
        pDeviceName = [NSString stringWithCString:strDeviceName encoding:NSUTF8StringEncoding];
        for(int i = 0; i < *plCount; i++)
        {
            pCameraList[i].strProduct = [NSString stringWithCString:cameraList[i].strProduct encoding:NSUTF8StringEncoding];
            pCameraList[i].strSerialNo = [NSString stringWithCString:cameraList[i].strSerialNo encoding:NSUTF8StringEncoding];
            pCameraList[i].strIPAddress = [NSString stringWithCString:cameraList[i].strIPAddress encoding:NSUTF8StringEncoding];
            pCameraList[i].strFramework = [NSString stringWithCString:cameraList[i].strFramework encoding:NSUTF8StringEncoding];
            pCameraList[i].bValid = cameraList[i].bValid;
        }
    }
    
    return result;
}

/**
 * @brief Disestablish a session between the camera and the computer.
 *
 * @param hCamera Camera handle.
 * @return API result.
 */
-(long) XSDK_Close:(XSDK_HANDLE) hCamera
{
    long result = pXSDK_Close(hCamera);
    
    return result;
}

/**
 * @brief Disestablish a session between the camera and the computer, and shut the camera down.
 *
 * @param hCamera Camera handle.
 * @return API result.
 */
-(long) XSDK_PowerOFF:(XSDK_HANDLE) hCamera
{
    long result = pXSDK_PowerOFF(hCamera);
    
    return result;
}

/**
 * @brief Establishes a session between the camera and the computer.
 *
 * @param pDevice Specify index or IP address.
 * @param phCamera Returns the camera handle when the function completes.
 * @param plCameraMode Returns a bitmap of camera features compatible with tethering operations.
 * @param pOption nullptr
 * @return API result.
 */
-(long) XSDK_OpenEx:(NSString*) pDevice
           phCamera:(XSDK_HANDLE*) phCamera
       plCameraMode:(long*) plCameraMode
            pOption:(void*) pOption
{
    char* strDevice = (char*)[pDevice UTF8String];
    
    long result = pXSDK_OpenEx(strDevice, phCamera, plCameraMode, pOption);
    
    return result;
}

/**
 * @brief Gets the detailed result of the last called function.
 *
 * @param hCamera Camera handle , phCamera can be set to NULL.
 * @param plAPICode The last called API code.
 * @param plERRCode See the ERROR CODES for details.
 * @return API result.
 */
-(long) XSDK_GetErrorNumber:(XSDK_HANDLE) hCamera
                  plAPICode:(long*) plAPICode
                  plERRCode:(long*) plERRCode
{
    long result = pXSDK_GetErrorNumber(hCamera, plAPICode, plERRCode);
    
    return result;
}

/**
 * @brief Gets version numbers in a string format.
 *
 * @param pVersionString Returns the SDK version as a string.
 * @return API result.
 */
-(long) XSDK_GetVersionString:(NSString*) pVersionString
{
    char* strVersionString = (char*)[pVersionString UTF8String];
    
    long result = pXSDK_GetVersionString(strVersionString);
    if(result == 0)
    {
        pVersionString = [NSString stringWithCString:strVersionString encoding:NSUTF8StringEncoding];
    }
    
    return result;
}

/**
 * @brief Gets details of the busy error when the plERRCode returned by a call to XSDK_GetErrorNumber is XSDK_ERRCODE_RUNNING_OTHER_FUNCTION.
 *
 * @param hCamera Camera handle , phCamera can be set to NULL.
 * @param plERRCode Returns the function currently running.
 * @return API result.
 */
-(long) XSDK_GetErrorDetails:(XSDK_HANDLE) hCamera
                   plERRCode:(long*) plERRCode
{
    long result = pXSDK_GetErrorDetails(hCamera, plERRCode);
    
    return result;
}

/**
 * @brief Gets information about the connected camera.
 *
 * @param hCamera Camera handle.
 * @param pDevInfo Returns camera information.
 * @return API result.
 */
-(long) XSDK_GetDeviceInfo:(XSDK_HANDLE) hCamera
                  pDevInfo:(DeviceInformation*) pDevInfo
{
    XSDK_DeviceInformation device;
    
    long result = pXSDK_GetDeviceInfo(hCamera, &device);
    if (result == 0)
    {
        pDevInfo.strVendor = [NSString stringWithCString:device.strVendor encoding:NSUTF8StringEncoding];
        pDevInfo.strManufacturer = [NSString stringWithCString:device.strManufacturer encoding:NSUTF8StringEncoding];
        pDevInfo.strProduct = [NSString stringWithCString:device.strProduct encoding:NSUTF8StringEncoding];
        pDevInfo.strFirmware = [NSString stringWithCString:device.strFirmware encoding:NSUTF8StringEncoding];
        pDevInfo.strDeviceType = [NSString stringWithCString:device.strDeviceType encoding:NSUTF8StringEncoding];
        pDevInfo.strSerialNo = [NSString stringWithCString:device.strSerialNo encoding:NSUTF8StringEncoding];
        pDevInfo.strFramework = [NSString stringWithCString:device.strFramework encoding:NSUTF8StringEncoding];
        pDevInfo.bDeviceId = device.bDeviceId;
        pDevInfo.strDeviceName = [NSString stringWithCString:device.strDeviceName encoding:NSUTF8StringEncoding];
        pDevInfo.strYNo = [NSString stringWithCString:device.strYNo encoding:NSUTF8StringEncoding];
    }
    
    return result;
}

/**
 * @brief Assigns a device-unique name to the camera.
 *
 * @param hCamera Camera handle.
 * @param pDeviceName A unique name. Up to 32 characters including the NULL terminator.
 * @return API result.
 */
-(long) XSDK_WriteDeviceName:(XSDK_HANDLE) hCamera
                 pDeviceName:(NSString*) pDeviceName
{
    char* strDeviceName = (char*)[pDeviceName UTF8String];
    
    long result = pXSDK_WriteDeviceName(hCamera, strDeviceName);
    if(result == 0)
    {
        pDeviceName = [NSString stringWithCString:strDeviceName encoding:NSUTF8StringEncoding];
    }
    
    return result;
}

/**
 * @brief Assigns a device-unique name to the camera.
 *
 * @param hCamera Camera handle.
 * @param pFirmwareVersion Returns the camera firmware version as a string.
 * @return API result.
 */
-(long) XSDK_GetFirmwareVersion:(XSDK_HANDLE) hCamera
               pFirmwareVersion:(NSString*) pFirmwareVersion
{
    char* strFirmwareVersion = (char*)[pFirmwareVersion UTF8String];
    
    long result = pXSDK_GetFirmwareVersion(hCamera, strFirmwareVersion);
    if(result == 0)
    {
        pFirmwareVersion = [NSString stringWithCString:strFirmwareVersion encoding:NSUTF8StringEncoding];
    }
    
    return result;
}

/**
 * @brief Gets lens information from the camera.
 *
 * @param hCamera Camera handle.
 * @param pLensInfo Returns lens information.
 * @return API result.
 */
-(long) XSDK_GetLensInfo:(XSDK_HANDLE) hCamera
               pLensInfo:(LensInformation*) pLensInfo
{
    XSDK_LensInformation lensInfo;
    
    long result = pXSDK_GetLensInfo(hCamera, &lensInfo);
    if (result == 0)
    {
        pLensInfo.strModel = [NSString stringWithCString:lensInfo.strModel encoding:NSUTF8StringEncoding];
        pLensInfo.strProductName = [NSString stringWithCString:lensInfo.strProductName encoding:NSUTF8StringEncoding];
        pLensInfo.strSerialNo = [NSString stringWithCString:lensInfo.strSerialNo encoding:NSUTF8StringEncoding];
        pLensInfo.lISCapability = lensInfo.lISCapability;
        pLensInfo.lMFCapability = lensInfo.lMFCapability;
        pLensInfo.lZoomPosCapability = lensInfo.lZoomPosCapability;
    }
    
    return result;
}

/**
 * @brief Gets the firmware version of the lens attached to the camera in a string format.
 *
 * @param hCamera Camera handle.
 * @param pLensVersion Returns the lens firmware version as a string.
 * @return API result.
 */
-(long) XSDK_GetLensVersion:(XSDK_HANDLE) hCamera
               pLensVersion:(NSString*) pLensVersion
{
    char* strLensVersion = (char*)[pLensVersion UTF8String];
    
    long result = pXSDK_GetLensVersion(hCamera, strLensVersion);
    if(result == 0)
    {
        pLensVersion = [NSString stringWithCString:strLensVersion encoding:NSUTF8StringEncoding];
    }
    
    return result;
}

/**
 * @brief Gets information about the connected camera and supported APIs by the camera.
 *
 * @param hCamera Camera handle.
 * @param pDevInfo Returns camera information.
 * @param plNumAPICode Returns the number of APICode supported.
 * @param plAPICode If not NULL, plAPICode will return a list of APICode supported.
 * @return API result.
 */
-(long) XSDK_GetDeviceInfoEx:(XSDK_HANDLE) hCamera
                    pDevInfo:(DeviceInformation*) pDevInfo
                plNumAPICode:(long*) plNumAPICode
                   plAPICode:(long*) plAPICode
{
    long result = XSDK_ERROR;
    XSDK_DeviceInformation device;
    
    if(plAPICode == NULL)
    {
        result = pXSDK_GetDeviceInfoEx(hCamera, &device, plNumAPICode, NULL);
        if (result == XSDK_COMPLETE)
        {
            pDevInfo.strVendor = [NSString stringWithCString:device.strVendor encoding:NSUTF8StringEncoding];
            pDevInfo.strManufacturer = [NSString stringWithCString:device.strManufacturer encoding:NSUTF8StringEncoding];
            pDevInfo.strProduct = [NSString stringWithCString:device.strProduct encoding:NSUTF8StringEncoding];
            pDevInfo.strFirmware = [NSString stringWithCString:device.strFirmware encoding:NSUTF8StringEncoding];
            pDevInfo.strDeviceType = [NSString stringWithCString:device.strDeviceType encoding:NSUTF8StringEncoding];
            pDevInfo.strSerialNo = [NSString stringWithCString:device.strSerialNo encoding:NSUTF8StringEncoding];
            pDevInfo.strFramework = [NSString stringWithCString:device.strFramework encoding:NSUTF8StringEncoding];
            pDevInfo.bDeviceId = device.bDeviceId;
            pDevInfo.strDeviceName = [NSString stringWithCString:device.strDeviceName encoding:NSUTF8StringEncoding];
            pDevInfo.strYNo = [NSString stringWithCString:device.strYNo encoding:NSUTF8StringEncoding];
        }
    }
    else
    {
        long list[*plNumAPICode];
        
        result = pXSDK_GetDeviceInfoEx(hCamera, &device, plNumAPICode, list);
        if (result == XSDK_COMPLETE)
        {
            pDevInfo.strVendor = [NSString stringWithCString:device.strVendor encoding:NSUTF8StringEncoding];
            pDevInfo.strManufacturer = [NSString stringWithCString:device.strManufacturer encoding:NSUTF8StringEncoding];
            pDevInfo.strProduct = [NSString stringWithCString:device.strProduct encoding:NSUTF8StringEncoding];
            pDevInfo.strFirmware = [NSString stringWithCString:device.strFirmware encoding:NSUTF8StringEncoding];
            pDevInfo.strDeviceType = [NSString stringWithCString:device.strDeviceType encoding:NSUTF8StringEncoding];
            pDevInfo.strSerialNo = [NSString stringWithCString:device.strSerialNo encoding:NSUTF8StringEncoding];
            pDevInfo.strFramework = [NSString stringWithCString:device.strFramework encoding:NSUTF8StringEncoding];
            pDevInfo.bDeviceId = device.bDeviceId;
            pDevInfo.strDeviceName = [NSString stringWithCString:device.strDeviceName encoding:NSUTF8StringEncoding];
            pDevInfo.strYNo = [NSString stringWithCString:device.strYNo encoding:NSUTF8StringEncoding];
            
            for(int i = 0; i < *plNumAPICode; i++)
            {
                plAPICode[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Queries supported operation modes.
 *
 * @param hCamera Camera handle.
 * @param plNumPriorityMode Returns the supported number of settings for XSDK_SetPriorityMode.
 * @param plPriorityMode If not NULL, plPriorityMode will return a list of the XSDK_SetPriorityMode settings supported.
 * @return API result.
 */
-(long) XSDK_CapPriorityMode:(XSDK_HANDLE) hCamera
           plNumPriorityMode:(long*) plNumPriorityMode
              plPriorityMode:(long*) plPriorityMode
{
    long result = XSDK_ERROR;
    
    if(plPriorityMode == NULL)
    {
        result = pXSDK_CapPriorityMode(hCamera, plNumPriorityMode, NULL);
    }
    else
    {
        long list[*plNumPriorityMode];
        
        result = pXSDK_CapPriorityMode(hCamera, plNumPriorityMode, list);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNumPriorityMode; i++)
            {
                plPriorityMode[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Sets the camera operation mode.
 *
 * @param hCamera Camera handle.
 * @param lPriorityMode The priority mode.
 * @return API result.
 */
-(long) XSDK_SetPriorityMode:(XSDK_HANDLE) hCamera
               lPriorityMode:(long) lPriorityMode
{
    long result = pXSDK_SetPriorityMode(hCamera, lPriorityMode);
    
    return result;
}

/**
 * @brief Gets the current camera operation mode.
 *
 * @param hCamera Camera handle.
 * @param pPriorityMode The priority mode.
 * @return API result.
 */
-(long) XSDK_GetPriorityMode:(XSDK_HANDLE) hCamera
              plPriorityMode:(long*) plPriorityMode
{
    long result = pXSDK_GetPriorityMode(hCamera, plPriorityMode);
    
    return result;
}

/**
 * @brief Queries supported release-related modes when the system is in PC priority mode.
 *
 * @param hCamera Camera handle.
 * @param plNumReleaseMode The number of supported XSDK_Release settings.
 * @param plReleaseMode If plReleaseMode is NULL, the function will return only plNumReleaseMode with the number of supported XSDK_Release lReleaseMode settings.
 * @return API result.
 */
-(long) XSDK_CapRelease:(XSDK_HANDLE) hCamera
       plNumReleaseMode:(long*) plNumReleaseMode
          plReleaseMode:(long*) plReleaseMode
{
    long result = XSDK_ERROR;
    
    if(plReleaseMode == NULL)
    {
        result = pXSDK_CapRelease(hCamera, plNumReleaseMode, NULL);
    }
    else
    {
        long list[*plNumReleaseMode];
        
        result = pXSDK_CapRelease(hCamera, plNumReleaseMode, list);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNumReleaseMode; i++)
            {
                plReleaseMode[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Triggers shutter release-related operations when the system is in PC priority mode.
 *
 * @param hCamera Camera handle.
 * @param lReleaseMode If plReleaseMode is NULL, the function will return only plNumReleaseMode with the number of supported XSDK_Release lReleaseMode settings.
 * @param pShotOpt Specifies the number of pictures to be taken per burst in burst photography modes and returns the number of pictures actually taken.
 * @param pStatus Sometimes returns AF status when the function is called for S1- and S2-related operations.
 *                  Calls to measure custom white balance return autoexposure status.
 * @return API result.
 */
-(long) XSDK_Release:(XSDK_HANDLE) hCamera
        lReleaseMode:(long) lReleaseMode
           plShotOpt:(long*) plShotOpt
          plAfStatus:(long*) plAfStatus
{
    long result = pXSDK_Release(hCamera, lReleaseMode, plShotOpt, plAfStatus);
    
    return result;
}

/**
 * @brief Queries supported release-related modes when the system is in CAMERA priority mode.
 *
 * @param hCamera Camera handle.
 * @param plNumReleaseMode The number of supported XSDK_Release settings.
 * @param plReleaseMode If pulReleaseMode is NULL, the function will return only plNumReleaseMode with the number of supported XSDK_ReleaseEx ulReleaseMode settings.
 *                      Otherwise it will return pulReleaseMode with a list of the XSDK_ReleaseEx ulReleaseMode settings supported.
 * @return API result.
 */
-(long) XSDK_CapReleaseEx:(XSDK_HANDLE) hCamera
         plNumReleaseMode:(long*) plNumReleaseMode
            plReleaseMode:(long*) plReleaseMode
{
    long result = XSDK_ERROR;
    
    if(plReleaseMode == NULL)
    {
        result = pXSDK_CapRelease(hCamera, plNumReleaseMode, NULL);
    }
    else
    {
        long list[*plNumReleaseMode];
        
        result = pXSDK_CapRelease(hCamera, plNumReleaseMode, list);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNumReleaseMode; i++)
            {
                plReleaseMode[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Triggers shutter release-related operations when the system is in camera priority mode.
 *
 * @param hCamera Camera handle.
 * @param lReleaseMode If pulReleaseMode is NULL, the function will return only plNumReleaseMode with the number of supported XSDK_ReleaseEx ulReleaseMode settings.
 *                      Otherwise it will return pulReleaseMode with a list of the XSDK_ReleaseEx ulReleaseMode settings supported.
 * @param pShotOpt Specifies the number of pictures to be taken per burst in burst photography modes and returns the number of pictures actually taken.
 * @param plAfStatus Sometimes returns AF status when the function is called for S1- and S2-related operations.
 *                  Calls to measure custom white balance return autoexposure status.
 * @return API result.
 */
-(long) XSDK_ReleaseEx:(XSDK_HANDLE) hCamera
          lReleaseMode:(long) lReleaseMode
             plShotOpt:(long*) plShotOpt
            plAfStatus:(long*) plAfStatus
{
    long result = pXSDK_ReleaseEx(hCamera, lReleaseMode, plShotOpt, plAfStatus);
    
    return result;
}

/**
 * @brief Gets the status of release operation.
 *
 * @param hCamera Camera handle.
 * @param plReleaseStatus Returns release status via bit ON/OFF.
 * @return API result.
 */
-(long) XSDK_GetReleaseStatus:(XSDK_HANDLE) hCamera
              plReleaseStatus:(long*) plReleaseStatus
{
    long result = pXSDK_GetReleaseStatus(hCamera, plReleaseStatus);
    
    return result;
}

/**
 * @brief Gets the status of the in-camera buffer.
 *
 * @param hCamera Camera handle.
 * @param pShootFrameNum Returns the number of frames shot (the number of captured images).
 * @param plTotalFrameNum Returns the total number of frames.
 * @return API result.
 */
-(long) XSDK_GetBufferCapacity:(XSDK_HANDLE) hCamera
               plShootFrameNum:(long*) plShootFrameNum
               plTotalFrameNum:(long*) plTotalFrameNum
{
    long result = pXSDK_GetBufferCapacity(hCamera, plShootFrameNum, plTotalFrameNum);
    
    return result;
}

/**
 * @brief Gets information from an image from the top of the in-camera buffer.
 *
 * @param hCamera Camera handle.
 * @param pImgInfo Returns the image information.
 * @return API result.
 */
-(long) XSDK_ReadImageInfo:(XSDK_HANDLE) hCamera
                  pImgInfo:(ImageInformation*) pImgInfo
{
    XSDK_ImageInformation imageInfo;
    
    long result = pXSDK_ReadImageInfo(hCamera, &imageInfo);
    if (result == 0)
    {
        pImgInfo.strInternalName = [NSString stringWithCString:imageInfo.strInternalName encoding:NSUTF8StringEncoding];
        pImgInfo.lFormat = imageInfo.lFormat;
        pImgInfo.lDataSize = imageInfo.lDataSize;
        pImgInfo.lImagePixHeight = imageInfo.lImagePixHeight;
        pImgInfo.lImagePixWidth = imageInfo.lImagePixWidth;
        pImgInfo.lImageBitDepth = imageInfo.lImageBitDepth;
        pImgInfo.lPreviewSize = imageInfo.lPreviewSize;
        pImgInfo.hImage = imageInfo.hImage;
    }
    
    return result;
}

/**
 * @brief Gets a low-resolution image of the image from the top of the in-camera buffer.
 *
 * @param hCamera Camera handle.
 * @param pData The read-image buffer.
 * @param lDataSize The number of bytes allocated for pData.
 * @return API result.
 */
-(long) XSDK_ReadPreview:(XSDK_HANDLE) hCamera
                   pData:(uint8_t*) pData
               lDataSize:(unsigned long) lDataSize
{
    long result = pXSDK_ReadPreview(hCamera, pData, lDataSize);
    
    return result;
}

/**
 * @brief Gets a captured image from the top of the in-camera buffer and deletes it from the buffer.
 *
 * @param hCamera Camera handle.
 * @param pData The read-image buffer.
 * @param lDataSize The number of bytes allocated for pData.
 * @return API result.
 */
-(long) XSDK_ReadImage:(XSDK_HANDLE) hCamera
                 pData:(uint8_t*) pData
             lDataSize:(unsigned long) lDataSize
{
    long result = pXSDK_ReadImage(hCamera, pData, lDataSize);
    
    return result;
}

/**
 * @brief Deletes a captured image from the top of the in-camera buffer.
 *
 * @param hCamera Camera handle.
 * @return API result.
 */
-(long) XSDK_DeleteImage:(XSDK_HANDLE) hCamera
{
    long result = pXSDK_DeleteImage(hCamera);
    
    return result;
}

/**
 * @brief Queries supported exposure modes (P/A/S/M) to set.
 *
 * @param hCamera Camera handle.
 * @param plNumAEMode Returns the number of supported XSDK_SetAEMode settings.
 * @param plAEMode If not NULL, plAEMode will return a list of the XSDK_SetAEMode settings supported.
 * @return API result.
 */
-(long) XSDK_CapAEMode:(XSDK_HANDLE) hCamera
           plNumAEMode:(long*) plNumAEMode
              plAEMode:(long*) plAEMode
{
    long result = XSDK_ERROR;
    
    if(plAEMode == NULL)
    {
        result = pXSDK_CapAEMode(hCamera, plNumAEMode, NULL);
    }
    else
    {
        long list[*plNumAEMode];
        
        result = pXSDK_CapAEMode(hCamera, plNumAEMode, list);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNumAEMode; i++)
            {
                plAEMode[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Sets the exposure mode setting.
 *
 * @param hCamera Camera handle.
 * @param lAEMode The exposure mode to which the camera will be set.
 * @return API result.
 */
-(long) XSDK_SetAEMode:(XSDK_HANDLE) hCamera
               lAEMode:(long) lAEMode
{
    long result = pXSDK_SetAEMode(hCamera, lAEMode);
    
    return result;
}

/**
 * @brief Sets the exposure mode setting.
 *
 * @param hCamera Camera handle.
 * @param plAEMode Returns the exposure mode.
 * @return API result.
 */
-(long) XSDK_GetAEMode:(XSDK_HANDLE) hCamera
              plAEMode:(long*) plAEMode
{
    long result = pXSDK_GetAEMode(hCamera, plAEMode);
    
    return result;
}

/**
 * @brief Queries supported shutter speeds to set.
 *
 * @param hCamera Camera handle.
 * @param plNumShutterSpeed Returns the number of supported XSDK_SetShutterSpeed settings.
 * @param plShutterSpeed If not NULL, plShutterSpeed will return a list of the XSDK_SetShutterSpeed settings supported.
 * @param plBulbCapable BULB Capable.
 * @return API result.
 */
-(long) XSDK_CapShutterSpeed:(XSDK_HANDLE) hCamera
           plNumShutterSpeed:(long*) plNumShutterSpeed
              plShutterSpeed:(long*) plShutterSpeed
               plBulbCapable:(long*) plBulbCapable
{
    long result = XSDK_ERROR;
    
    if(plShutterSpeed == NULL)
    {
        result = pXSDK_CapShutterSpeed(hCamera, plNumShutterSpeed, NULL, plBulbCapable);
    }
    else
    {
        long list[*plNumShutterSpeed];
        
        result = pXSDK_CapShutterSpeed(hCamera, plNumShutterSpeed, list, plBulbCapable);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNumShutterSpeed; i++)
            {
                plShutterSpeed[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Sets the shutter speed value.
 *
 * @param hCamera Camera handle.
 * @param lShutterSpeed The shutter speed to which the camera will be set.
 * @param lBulb BULB Capable.
 * @return API result.
 */
-(long) XSDK_SetShutterSpeed:(XSDK_HANDLE) hCamera
               lShutterSpeed:(long) lShutterSpeed
                       lBulb:(long) lBulb
{
    long result = pXSDK_SetShutterSpeed(hCamera, lShutterSpeed, lBulb);
    
    return result;
}

/**
 * @brief Gets the shutter speed setting.
 *
 * @param hCamera Camera handle.
 * @param plShutterSpeed Returns the shutter speed.
 * @param plBulb Returns  the shutter speed setting is BULB or not.
 * @return API result.
 */
-(long) XSDK_GetShutterSpeed:(XSDK_HANDLE) hCamera
              plShutterSpeed:(long*) plShutterSpeed
                      plBulb:(long*) plBulb
{
    long result = pXSDK_GetShutterSpeed(hCamera, plShutterSpeed, plBulb);
    
    return result;
}

/**
 * @brief Queries supported exposure compensations to set.
 *
 * @param hCamera Camera handle.
 * @param plNumExposureBias Returns the number of supported XSDK_SetExposureBias settings.
 * @param plExposureBias If not NULL, plExposureBias will return a list of the XSDK_SetExposureBias settings supported.
 * @return API result.
 */
-(long) XSDK_CapExposureBias:(XSDK_HANDLE) hCamera
           plNumExposureBias:(long*) plNumExposureBias
              plExposureBias:(long*) plExposureBias
{
    long result = XSDK_ERROR;
    
    if(plExposureBias == NULL)
    {
        result = pXSDK_CapExposureBias(hCamera, plNumExposureBias, NULL);
    }
    else
    {
        long list[*plNumExposureBias];
        
        result = pXSDK_CapExposureBias(hCamera, plNumExposureBias, list);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNumExposureBias; i++)
            {
                plExposureBias[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Sets the exposure compensation value.
 *
 * @param hCamera Camera handle.
 * @param lExposureBias The value to which exposure compensation will be set.
 * @return API result.
 */
-(long) XSDK_SetExposureBias:(XSDK_HANDLE) hCamera
               lExposureBias:(long) lExposureBias
{
    long result = pXSDK_SetExposureBias(hCamera, lExposureBias);
    
    return result;
}

/**
 * @brief Gets the exposure compensation setting.
 *
 * @param hCamera Camera handle.
 * @param plExposureBias Returns the current exposure compensation value.
 * @return API result.
 */

-(long) XSDK_GetExposureBias:(XSDK_HANDLE) hCamera
              plExposureBias:(long*) plExposureBias
{
    long result = pXSDK_GetExposureBias(hCamera, plExposureBias);
    
    return result;
}

/**
 * @brief Queries supported ISO sensitivities to set.
 *
 * @param hCamera Camera handle.
 * @param plNumSensitivity Returns the number of supported settings for XSDK_SetSensitivity.
 * @param plSensitivity If not NULL, plSensitivity will return a list of the XSDK_SetSensitivity settings supported.
 * @return API result.
 */
-(long) XSDK_CapSensitivity:(XSDK_HANDLE) hCamera
           plNumSensitivity:(long*) plNumSensitivity
              plSensitivity:(long*) plSensitivity
{
    long result = XSDK_ERROR;
    
    if(plSensitivity == NULL)
    {
        result = pXSDK_CapSensitivity(hCamera, plNumSensitivity, NULL);
    }
    else
    {
        long list[*plNumSensitivity];
        
        result = pXSDK_CapSensitivity(hCamera, plNumSensitivity, list);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNumSensitivity; i++)
            {
                plSensitivity[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Sets the ISO sensitivity value.
 *
 * @param hCamera Camera handle.
 * @param lSensitivity The value to which sensitivity will be set.
 * @return API result.
 */
-(long) XSDK_SetSensitivity:(XSDK_HANDLE) hCamera
               lSensitivity:(long) lSensitivity
{
    long result = pXSDK_SetSensitivity(hCamera, lSensitivity);
    
    return result;
}

/**
 * @brief Gets the ISO sensitivity setting.
 *
 * @param hCamera Camera handle.
 * @param plSensitivity Returns the current value for ISO sensitivity.
 * @return API result.
 */
-(long) XSDK_GetSensitivity:(XSDK_HANDLE) hCamera
              plSensitivity:(long*) plSensitivity
{
    long result = pXSDK_GetSensitivity(hCamera, plSensitivity);
    
    return result;
}

/**
 * @brief Queries supported dynamic ranges to set.
 *
 * @param hCamera Camera handle.
 * @param plNumDynamicRange Returns the number of supported XSDK_SetDynamicRange settings.
 * @param plDynamicRange If not NULL, plDynamicRange will return a list of the XSDK_SetDynamicRange settings supported.
 * @return API result.
 */
-(long) XSDK_CapDynamicRange:(XSDK_HANDLE) hCamera
           plNumDynamicRange:(long*) plNumDynamicRange
              plDynamicRange:(long*) plDynamicRange
{
    long result = XSDK_ERROR;
    
    if(plDynamicRange == NULL)
    {
        result = pXSDK_CapDynamicRange(hCamera, plNumDynamicRange, NULL);
    }
    else
    {
        long list[*plNumDynamicRange];
        
        result = pXSDK_CapDynamicRange(hCamera, plNumDynamicRange, list);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNumDynamicRange; i++)
            {
                plDynamicRange[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Sets the dynamic range value.
 *
 * @param hCamera Camera handle.
 * @param lDynamicRange The value to which dynamic range will be set.
 * @return API result.
 */
-(long) XSDK_SetDynamicRange:(XSDK_HANDLE) hCamera
               lDynamicRange:(long) lDynamicRange
{
    long result = pXSDK_SetDynamicRange(hCamera, lDynamicRange);
    
    return result;
}

/**
 * @brief Gets the dynamic range setting.
 *
 * @param hCamera Camera handle.
 * @param plDynamicRange Returns the dynamic range.
 * @return API result.
 */
-(long) XSDK_GetDynamicRange:(XSDK_HANDLE) hCamera
              plDynamicRange:(long*) plDynamicRange
{
    long result = pXSDK_GetDynamicRange(hCamera, plDynamicRange);
    
    return result;
}

/**
 * @brief Queries supported metering modes to set.
 *
 * @param hCamera Camera handle.
 * @param plNumMeteringMode Returns the number of supported XSDK_SetMeteringMode settings.
 * @param plMeteringMode If not NULL, plMeteringMode will return a list of the XSDK_SetMeteringMode settings supported.
 * @return API result.
 */
-(long) XSDK_CapMeteringMode:(XSDK_HANDLE) hCamera
           plNumMeteringMode:(long*) plNumMeteringMode
              plMeteringMode:(long*) plMeteringMode
{
    long result = XSDK_ERROR;
    
    if(plMeteringMode == NULL)
    {
        result = pXSDK_CapMeteringMode(hCamera, plNumMeteringMode, NULL);
    }
    else
    {
        long list[*plNumMeteringMode];
        
        result = pXSDK_CapMeteringMode(hCamera, plNumMeteringMode, list);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNumMeteringMode; i++)
            {
                plMeteringMode[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Sets the metering mode.
 *
 * @param hCamera Camera handle.
 * @param lMeteringMode The metering mode
 * @return API result.
 */
-(long) XSDK_SetMeteringMode:(XSDK_HANDLE) hCamera
               lMeteringMode:(long) lMeteringMode
{
    long result = pXSDK_SetMeteringMode(hCamera, lMeteringMode);
    
    return result;
}

/**
 * @brief Gets the metering mode setting.
 *
 * @param hCamera Camera handle.
 * @param plMeteringMode The metering mode
 * @return API result.
 */
-(long) XSDK_GetMeteringMode:(XSDK_HANDLE) hCamera
              plMeteringMode:(long*) plMeteringMode
{
    long result = pXSDK_GetMeteringMode(hCamera, plMeteringMode);
    
    return result;
}

/**
 * @brief Queries supported zoom positions to set.
 *
 * @param hCamera Camera handle.
 * @param plNumZoomPos Returns the number of supported zoom positions.
 * @param plZoomPos If not NULL, plZoomPos returns a list of the zoom positions supported.
 * @param plFocusLength If not NULL, plFocalLength returns a list of the focal length positions supported. The values are 100 times the actual focal lengths.
 * @param pl35mmFocusLength If not NULL, pl35mmFocalLength returns a list of the 35 mm-equivalent focal length positions supported. The values are 100 times the actual focal lengths.
 * @return API result.
 */
-(long) XSDK_CapLensZoomPos:(XSDK_HANDLE) hCamera
               plNumZoomPos:(long*) plNumZoomPos
                  plZoomPos:(long*) plZoomPos
              plFocusLength:(long*) plFocusLength
          pl35mmFocusLength:(long*) pl35mmFocusLength
{
    long result = XSDK_ERROR;
    
    if(plZoomPos == NULL && plFocusLength == NULL && pl35mmFocusLength == NULL)
    {
        result = pXSDK_CapLensZoomPos(hCamera, plNumZoomPos, NULL, NULL, NULL);
    }
    else
    {
        long list[*plNumZoomPos];
        long list2[*plNumZoomPos];
        long list3[*plNumZoomPos];
        
        result = pXSDK_CapLensZoomPos(hCamera, plNumZoomPos, list, list2, list3);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNumZoomPos; i++)
            {
                plZoomPos[i] = list[i];
                plFocusLength[i] = list[i];
                pl35mmFocusLength[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Sets the zoom position.
 *
 * @param hCamera Camera handle.
 * @param lZoomPos The zoom position, in steps.
 * @return API result.
 */
-(long) XSDK_SetLensZoomPos:(XSDK_HANDLE) hCamera
                   lZoomPos:(long) lZoomPos
{
    long result = pXSDK_SetLensZoomPos(hCamera, lZoomPos);
    
    return result;
}

/**
 * @brief Gets the zoom position setting.
 *
 * @param hCamera Camera handle.
 * @param plZoomPos The zoom position, in steps.
 * @return API result.
 */
-(long) XSDK_GetLensZoomPos:(XSDK_HANDLE) hCamera
                  plZoomPos:(long*) plZoomPos
{
    long result = pXSDK_GetLensZoomPos(hCamera, plZoomPos);
    
    return result;
}

/**
 * @brief Queries supported aperture values to set.
 *
 * @param hCamera Camera handle.
 * @param lZoomPos The zoom position retrieved using “XSDK_CapLensZoomPos”.
 * @param plNumAperture Returns the number of XSDK_SetAperture settings available at zoom position lZoomPos.
 * @param plFNumber If not NULL, plFNumber returns a list of the XSDK_SetAperture settings available at zoom position lZoomPos.
 * @return API result.
 */
-(long) XSDK_CapAperture:(XSDK_HANDLE) hCamera
                lZoomPos:(long) lZoomPos
           plNumAperture:(long*) plNumAperture
               plFNumber:(long*) plFNumber
{
    long result = XSDK_ERROR;
    
    if(plFNumber == NULL)
    {
        result = pXSDK_CapAperture(hCamera, lZoomPos, plNumAperture, NULL);
    }
    else
    {
        long list[*plNumAperture];
        
        result = pXSDK_CapAperture(hCamera, lZoomPos, plNumAperture, list);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNumAperture; i++)
            {
                plFNumber[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Sets the aperture value.
 *
 * @param hCamera Camera handle.
 * @param lFNumber The aperture value, expressed as the F number multiplied by 100.
 * @return API result.
 */
-(long) XSDK_SetAperture:(XSDK_HANDLE) hCamera
                lFNumber:(long) lFNumber
{
    long result = pXSDK_SetAperture(hCamera, lFNumber);
    
    return result;
}

/**
 * @brief Gets the aperture setting.
 *
 * @param hCamera Camera handle.
 * @param plFNumber Receive the aperture in F number.
 *                  The value is hundredfold value of the actual aperture F number.
 * @return API result.
 */
-(long) XSDK_GetAperture:(XSDK_HANDLE) hCamera
               plFNumber:(long*) plFNumber
{
    long result = pXSDK_GetAperture(hCamera, plFNumber);
    
    return result;
}

/**
 * @brief Queries supported white-balance modes to set.
 *
 * @param hCamera Camera handle.
 * @param plNumWBMode Returns the number of supported XSDK_SetWBMode settings.
 * @param plWBMode If not NULL, plWBMode returns a list of the XSDK_SetWBMode settings supported.
 * @return API result.
 */
-(long) XSDK_CapWBMode:(XSDK_HANDLE) hCamera
           plNumWBMode:(long*) plNumWBMode
              plWBMode:(long*) plWBMode
{
    long result = XSDK_ERROR;
    
    if(plWBMode == NULL)
    {
        result = pXSDK_CapWBMode(hCamera, plNumWBMode, NULL);
    }
    else
    {
        long list[*plNumWBMode];
        
        result = pXSDK_CapWBMode(hCamera, plNumWBMode, list);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNumWBMode; i++)
            {
                plWBMode[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Sets the white-balance mode.
 *
 * @param hCamera Camera handle.
 * @param lWBMode The white-balance mode.
 * @return API result.
 */
-(long) XSDK_SetWBMode:(XSDK_HANDLE) hCamera
               lWBMode:(long) lWBMode
{
    long result = pXSDK_SetWBMode(hCamera, lWBMode);
    
    return result;
}

/**
 * @brief Gets the white-balance mode setting.
 *
 * @param hCamera Camera handle.
 * @param plWBMode The current white-balance mode.
 * @return API result.
 */
-(long) XSDK_GetWBMode:(XSDK_HANDLE) hCamera
              plWBMode:(long*) plWBMode
{
    long result = pXSDK_GetWBMode(hCamera, plWBMode);
    
    return result;
}

/**
 * @brief Queries supported color temperatures to set available when WBMode=ColorTemperature.
 *
 * @param hCamera Camera handle.
 * @param plNumWBColorTemp Returns the number of supported XSDK_SetWBColorTemp settings.
 * @param plWBColorTemp If not NULL, plWBColorTemp returns a list of the XSDK_SetWBColorTemp settings supported.
 * @return API result.
 */
-(long) XSDK_CapWBColorTemp:(XSDK_HANDLE) hCamera
           plNumWBColorTemp:(long*) plNumWBColorTemp
              plWBColorTemp:(long*) plWBColorTemp
{
    long result = XSDK_ERROR;
    
    if(plWBColorTemp == NULL)
    {
        result = pXSDK_CapWBColorTemp(hCamera, plNumWBColorTemp, NULL);
    }
    else
    {
        long list[*plNumWBColorTemp];
        
        result = pXSDK_CapWBColorTemp(hCamera, plNumWBColorTemp, list);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNumWBColorTemp; i++)
            {
                plWBColorTemp[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Sets the color temperature value for WBMode=ColorTemperature.
 *
 * @param hCamera Camera handle.
 * @param lColorTemp The color temperature, in degrees Kelvin.
 * @return API result.
 */
-(long) XSDK_SetWBColorTemp:(XSDK_HANDLE) hCamera
                 lColorTemp:(long) lColorTemp
{
    long result = pXSDK_SetWBColorTemp(hCamera, lColorTemp);
    
    return result;
}

/**
 * @brief Gets the color temperature setting for WBMode=ColorTemperature.
 *
 * @param hCamera Camera handle.
 * @param plColorTemp Returns the current color temperature, in degrees Kelvin.
 * @return API result.
 */
-(long) XSDK_GetWBColorTemp:(XSDK_HANDLE) hCamera
                plColorTemp:(long*) plColorTemp
{
    long result = pXSDK_GetWBColorTemp(hCamera, plColorTemp);
    
    return result;
}

/**
 * @brief Queries supported media recording control modes to set.
 *
 * @param hCamera Camera handle.
 * @param plNumMediaRecord Returns the number of supported XSDK_SetMediaRecord settings.
 * @param plMediaRecord If not NULL, plMediaRecord returns a list of the XSDK_SetMediaRecord settings supported.
 * @return API result.
 */
-(long) XSDK_CapMediaRecord:(XSDK_HANDLE) hCamera
           plNumMediaRecord:(long*) plNumMediaRecord
              plMediaRecord:(long*) plMediaRecord
{
    long result = XSDK_ERROR;
    
    if(plMediaRecord == NULL)
    {
        result = pXSDK_CapMediaRecord(hCamera, plNumMediaRecord, NULL);
    }
    else
    {
        long list[*plNumMediaRecord];
        
        result = pXSDK_CapMediaRecord(hCamera, plNumMediaRecord, list);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNumMediaRecord; i++)
            {
                plMediaRecord[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Sets the media recording control modes for the tethering operation.
 *
 * @param hCamera Camera handle.
 * @param lMediaRecord The media recording control mode.
 * @return API result.
 */
-(long) XSDK_SetMediaRecord:(XSDK_HANDLE) hCamera
               lMediaRecord:(long) lMediaRecord
{
    long result = pXSDK_SetMediaRecord(hCamera, lMediaRecord);
    
    return result;
}

/**
 * @brief Gets the media recording control modes setting for the tethering operation.
 *
 * @param hCamera Camera handle.
 * @param plMediaRecord The current media recording control mode.
 * @return API result.
 */
-(long) XSDK_GetMediaRecord:(XSDK_HANDLE) hCamera
              plMediaRecord:(long*) plMediaRecord
{
    long result = pXSDK_GetMediaRecord(hCamera, plMediaRecord);
    
    return result;
}

/**
 * @brief Queries supported operation modes to set.
 *
 * @param hCamera Camera handle.
 * @param plNumStandbyMode Returns the number of supported XSDK_SetForceMode settings.
 * @param plForceMode If not NULL, plForceMode returns a list of the XSDK_SetForceMode settings supported.
 * @return API result.
 */
-(long) XSDK_CapForceMode:(XSDK_HANDLE) hCamera
           plNumForceMode:(long*) plNumForceMode
              plForceMode:(long*) plForceMode
{
    long result = XSDK_ERROR;
    
    if(plForceMode == NULL)
    {
        result = pXSDK_CapForceMode(hCamera, plNumForceMode, NULL);
    }
    else
    {
        long list[*plNumForceMode];
        
        result = pXSDK_CapForceMode(hCamera, plNumForceMode, list);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNumForceMode; i++)
            {
                plForceMode[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Forcibly changes the operating mode to SHOOTING MODE.
 *
 * @param hCamera Camera handle.
 * @param lForceMode The operating mode to SHOOTING MODE.
 * @return API result.
 */
-(long) XSDK_SetForceMode:(XSDK_HANDLE) hCamera
               lForceMode:(long) lForceMode
{
    long result = pXSDK_SetForceMode(hCamera, lForceMode);

    return result;
}

/**
 * @brief Queries supported drive modes.
 *
 * @param hCamera Camera handle.
 * @param plNum Returns the number of XSDK_SetDriveMode settings supported.
 * @param pFunctionMode If plDriveMode is NULL, the function will return only plNumDriveMode with the number of supported XSDK_SetDriveMode settings.
 * @return API result.
 */
-(long) XSDK_CapDriveMode:(XSDK_HANDLE) hCamera
                    plNum:(long*) plNum
            pFunctionMode:(long*) pFunctionMode
{
    long result = XSDK_ERROR;
    
    if(pFunctionMode == NULL)
    {
        result = pXSDK_CapDriveMode(hCamera, plNum, NULL);
    }
    else
    {
        long list[*plNum];
        
        result = pXSDK_CapDriveMode(hCamera, plNum, list);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNum; i++)
            {
                pFunctionMode[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Sets the camera drive mode.
 *
 * @param hCamera Camera handle.
 * @param functionMode The drive mode.
 * @return API result.
 */
-(long) XSDK_SetDriveMode:(XSDK_HANDLE) hCamera
             functionMode:(long) functionMode
{
    long result = pXSDK_SetDriveMode(hCamera, functionMode);
    
    return result;
}

/**
 * @brief Gets the current camera drive mode.
 *
 * @param hCamera Camera handle.
 * @param pFunctionMode Drive Mode
 * @return API result.
 */
-(long) XSDK_GetDriveMode:(XSDK_HANDLE) hCamera
            pFunctionMode:(long*) pFunctionMode
{
    long result = pXSDK_GetDriveMode(hCamera, pFunctionMode);
    
    return result;
}

/**
 * @brief Queries supported camera MODES.
 *
 * @param hCamera Camera handle.
 * @param plNum Returns the number of XSDK_SetMode settings supported.
 * @param plMode If plMode is NULL, the function will return only plNumMode with the number of supported XSDK_SetMode settings.
 * @return API result.
 */
-(long) XSDK_CapMode:(XSDK_HANDLE) hCamera
               plNum:(long*) plNum
              plMode:(long*) plMode
{
    long result = XSDK_ERROR;
    
    if(plMode == NULL)
    {
        result = pXSDK_CapMode(hCamera, plNum, NULL);
    }
    else
    {
        long list[*plNum];
        
        result = pXSDK_CapMode(hCamera, plNum, list);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNum; i++)
            {
                plMode[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Sets the camera MODE.
 *
 * @param hCamera Camera handle.
 * @param lMode The mode.
 * @return API result.
 */
-(long) XSDK_SetMode:(XSDK_HANDLE) hCamera
               lMode:(long) lMode
{
    long result = pXSDK_SetMode(hCamera, lMode);
    
    return result;
}

/**
 * @brief Gets the current camera MODE.
 *
 * @param hCamera Camera handle.
 * @param plMode The mode.
 * @return API result.
 */
-(long) XSDK_GetMode:(XSDK_HANDLE) hCamera
              plMode:(long*) plMode
{
    long result = pXSDK_GetMode(hCamera, plMode);
    
    return result;
}

/**
 * @brief Restore camera backup settings.
 *
 * @param hCamera Camera handle.
 * @param lSize The size of the backup data.
 * @param pBackup The backup data.
 * @return API result.
 */
-(long) XSDK_SetBackupSettings:(XSDK_HANDLE) hCamera
                         lSize:(long) lSize
                       pBackup:(uint8_t*) pBackup
{
    long result = pXSDK_SetBackupSettings(hCamera, lSize, pBackup);
    
    return result;
}

/**
 * @brief Backup camera settings.
 *
 * @param hCamera Camera handle.
 * @param plSize The size of the backup data.
 * @param pBackup Camera setting data.
 * @return API result.
 */
-(long) XSDK_GetBackupSettings:(XSDK_HANDLE) hCamera
                        plSize:(long*) plSize
                       pBackup:(uint8_t*) pBackup
{
    long result = pXSDK_GetBackupSettings(hCamera, plSize, pBackup);
    
    return result;
}

/**
 * @brief Gets the current video recording status.
 *
 * @param hCamera Camera handle.
 * @param plStatus The video recording status.
 * @return API result.
 */
-(long) XSDK_GetRecordingStatus:(XSDK_HANDLE) hCamera
                       plStatus:(long*) plStatus
{
    long result = pXSDK_GetRecordingStatus(hCamera, plStatus);
    
    return result;
}

/**
 * @brief Queries supported shutter speeds to set in movie mode.
 *
 * @param hCamera Camera handle.
 * @param plNumShutterSpeed Returns the number of supported XSDK_SetMovieShutterSpeed settings.
 * @param pllShutterSpeed If not NULL, pllShutterSpeed will return a list of the XSDK_SetMovieShutterSpeed settings supported.
 * @return API result.
 */
-(long) XSDK_CapMovieShutterSpeed:(XSDK_HANDLE) hCamera
                plNumShutterSpeed:(long*) plNumShutterSpeed
                  pllShutterSpeed:(long long*) pllShutterSpeed
{
    long result = XSDK_ERROR;
    
    if(pllShutterSpeed == NULL)
    {
        result = pXSDK_CapMovieShutterSpeed(hCamera, plNumShutterSpeed, NULL);
    }
    else
    {
        long long list[*plNumShutterSpeed];
        
        result = pXSDK_CapMovieShutterSpeed(hCamera, plNumShutterSpeed, list);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNumShutterSpeed; i++)
            {
                pllShutterSpeed[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Sets the shutter speed value in movie mode.
 *
 * @param hCamera Camera handle.
 * @param llShutterSpeed The value to which shutter speed will be set.
 * @return API result.
 */
-(long) XSDK_SetMovieShutterSpeed:(XSDK_HANDLE) hCamera
                   llShutterSpeed:(long long) llShutterSpeed
{
    long result = pXSDK_SetMovieShutterSpeed(hCamera, llShutterSpeed);
    
    return result;
}

/**
 * @brief Gets the shutter speed setting in movie mode.
 *
 * @param hCamera Camera handle.
 * @param pllShutterSpeed Returns the current value for shutter speed.
 * @return API result.
 */
-(long) XSDK_GetMovieShutterSpeed:(XSDK_HANDLE) hCamera
                  pllShutterSpeed:(long long*) pllShutterSpeed
{
    long result = pXSDK_GetMovieShutterSpeed(hCamera, pllShutterSpeed);
    
    return result;
}

/**
 * @brief Returns the current value for shutter speed.
 *
 * @param hCamera Camera handle.
 * @param plNumExposureBias Returns the number of supported XSDK_SetMovieExposureBias settings.
 * @param plExposureBias If not NULL, plExposureBias will return a list of the XSDK_SetMovieExposureBias settings supported.
 * @return API result.
 */
-(long) XSDK_CapMovieExposureBias:(XSDK_HANDLE) hCamera
                plNumExposureBias:(long*) plNumExposureBias
                   plExposureBias:(long*) plExposureBias
{
    long result = XSDK_ERROR;
    
    if(plExposureBias == NULL)
    {
        result = pXSDK_CapMovieExposureBias(hCamera, plNumExposureBias, NULL);
    }
    else
    {
        long list[*plNumExposureBias];
        
        result = pXSDK_CapMovieExposureBias(hCamera, plNumExposureBias, list);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNumExposureBias; i++)
            {
                plExposureBias[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Sets the exposure compensation value in movie mode.
 *
 * @param hCamera Camera handle.
 * @param lExposureBias The value to which exposure compensation will be set.
 * @return API result.
 */
-(long) XSDK_SetMovieExposureBias:(XSDK_HANDLE) hCamera
                    lExposureBias:(long) lExposureBias
{
    long result = pXSDK_SetMovieExposureBias(hCamera, lExposureBias);
    
    return result;
}

/**
 * @brief Gets the exposure compensation setting in movie mode.
 *
 * @param hCamera Camera handle.
 * @param plExposureBias Returns the current exposure compensation value.
 * @return API result.
 */
-(long) XSDK_GetMovieExposureBias:(XSDK_HANDLE) hCamera
                   plExposureBias:(long*) plExposureBias
{
    long result = pXSDK_GetMovieExposureBias(hCamera, plExposureBias);
    
    return result;
}

/**
 * @brief Queries supported ISO sensitivities to set in movie mode.
 *
 * @param hCamera Camera handle.
 * @param plNumSensitivity Returns the number of supported XSDK_SetMovieSensitivity settings.Returns the number of supported XSDK_SetMovieSensitivity settings.
 * @param plSensitivity If not NULL, plSensitivity will return a list of the XSDK_SetMovieSensitivity settings supported.
 * @return API result.
 */
-(long) XSDK_CapMovieSensitivity:(XSDK_HANDLE) hCamera
                plNumSensitivity:(long*) plNumSensitivity
                   plSensitivity:(long*) plSensitivity
{
    long result = XSDK_ERROR;
    
    if(plSensitivity == NULL)
    {
        result = pXSDK_CapMovieSensitivity(hCamera, plNumSensitivity, NULL);
    }
    else
    {
        long list[*plNumSensitivity];
        
        result = pXSDK_CapMovieSensitivity(hCamera, plNumSensitivity, list);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNumSensitivity; i++)
            {
                plSensitivity[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Sets the ISO sensitivity value in movie mode.
 *
 * @param hCamera Camera handle.
 * @param lSensitivity The value to which ISO sensitivity will be set.
 * @return API result.
 */
-(long) XSDK_SetMovieSensitivity:(XSDK_HANDLE) hCamera
                    lSensitivity:(long) lSensitivity
{
    long result = pXSDK_SetMovieSensitivity(hCamera, lSensitivity);
    
    return result;
}

/**
 * @brief Gets the ISO sensitivity setting in movie mode.
 *
 * @param hCamera Camera handle.
 * @param plSensitivity Returns the current value for ISO sensitivity.
 * @return API result.
 */

-(long) XSDK_GetMovieSensitivity:(XSDK_HANDLE) hCamera
                   plSensitivity:(long*) plSensitivity
{
    long result = pXSDK_GetMovieSensitivity(hCamera, plSensitivity);
    
    return result;
}

/**
 * @brief Queries supported aperture values to set in movie mode.
 *
 * @param hCamera Camera handle.
 * @param plNumAperture Returns the number of supported XSDK_SetMovieAperture settings.
 * @param plFNumber If not NULL, plAperture will return a list of the XSDK_SetMovieAperture settings supported.
 * @return API result.
 */
-(long) XSDK_CapMovieAperture:(XSDK_HANDLE) hCamera
                plNumAperture:(long*) plNumAperture
                    plFNumber:(long*) plFNumber
{
    long result = XSDK_ERROR;
    
    if(plFNumber == NULL)
    {
        result = pXSDK_CapMovieAperture(hCamera, plNumAperture, NULL);
    }
    else
    {
        long list[*plNumAperture];
        
        result = pXSDK_CapMovieAperture(hCamera, plNumAperture, list);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNumAperture; i++)
            {
                plFNumber[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Sets the aperture value in movie mode.
 *
 * @param hCamera Camera handle.
 * @param lFNumber The value to which aperture will be set.
 *                  The aperture values returned are 100 times the actual F numbers.
 * @return API result.
 */
-(long) XSDK_SetMovieAperture:(XSDK_HANDLE) hCamera
                     lFNumber:(long) lFNumber
{
    long result = pXSDK_SetMovieAperture(hCamera, lFNumber);
    
    return result;
}

/**
 * @brief Gets the aperture setting in movie mode.
 *
 * @param hCamera Camera handle.
 * @param plFNumber Returns the current value for aperture.
 * @return API result.
 */
-(long) XSDK_GetMovieAperture:(XSDK_HANDLE) hCamera
                    plFNumber:(long*) plFNumber
{
    long result = pXSDK_GetMovieAperture(hCamera, plFNumber);
    
    return result;
}

/**
 * @brief Queries supported dynamic ranges to set in movie mode.
 *
 * @param hCamera Camera handle.
 * @param plNumDynamicRange Returns the number of supported XSDK_SetMovieDynamicRange settings.
 * @param plDynamicRange If not NULL, plDynamicRange returns a list of the XSDK_SetWBMode settings supported.
 * @return API result.
 */
-(long) XSDK_CapMovieDynamicRange:(XSDK_HANDLE) hCamera
                plNumDynamicRange:(long*) plNumDynamicRange
                   plDynamicRange:(long*) plDynamicRange
{
    long result = XSDK_ERROR;
    
    if(plDynamicRange == NULL)
    {
        result = pXSDK_CapMovieDynamicRange(hCamera, plNumDynamicRange, NULL);
    }
    else
    {
        long list[*plNumDynamicRange];
        
        result = pXSDK_CapMovieDynamicRange(hCamera, plNumDynamicRange, list);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNumDynamicRange; i++)
            {
                plDynamicRange[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Sets the dynamic range value in movie mode.
 *
 * @param hCamera Camera handle.
 * @param lDynamicRange The value to which dynamic range will be set.
 * @return API result.
 */
-(long) XSDK_SetMovieDynamicRange:(XSDK_HANDLE) hCamera
                    lDynamicRange:(long) lDynamicRange
{
    long result = pXSDK_SetMovieDynamicRange(hCamera, lDynamicRange);
    
    return result;
}

/**
 * @brief Gets the dynamic range setting in movie mode.
 *
 * @param hCamera Camera handle.
 * @param plDynamicRange Returns the current value for dynamic range.
 * @return API result.
 */
-(long) XSDK_GetMovieDynamicRange:(XSDK_HANDLE) hCamera
                   plDynamicRange:(long*) plDynamicRange
{
    long result = pXSDK_GetMovieDynamicRange(hCamera, plDynamicRange);
    
    return result;
}

/**
 * @brief Queries supported metering modes to set in movie mode.
 *
 * @param hCamera Camera handle.
 * @param plNumMeteringMode Returns the number of supported XSDK_SetMovieMeteringMode settings.
 * @param plMeteringMode If not NULL, plMeteringMode returns a list of the XSDK_SetWBMode settings supported.
 * @return API result.
 */
-(long) XSDK_CapMovieMeteringMode:(XSDK_HANDLE) hCamera
                plNumMeteringMode:(long*) plNumMeteringMode
                   plMeteringMode:(long*) plMeteringMode
{
    long result = XSDK_ERROR;
    
    if(plMeteringMode == NULL)
    {
        result = pXSDK_CapMovieMeteringMode(hCamera, plNumMeteringMode, NULL);
    }
    else
    {
        long list[*plNumMeteringMode];
        
        result = pXSDK_CapMovieMeteringMode(hCamera, plNumMeteringMode, list);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNumMeteringMode; i++)
            {
                plMeteringMode[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Sets the metering mode in movie mode.
 *
 * @param hCamera Camera handle.
 * @param lMeteringMode The value to which metering mode will be set.
 * @return API result.
 */
-(long) XSDK_SetMovieMeteringMode:(XSDK_HANDLE) hCamera
                    lMeteringMode:(long) lMeteringMode
{
    long result = pXSDK_SetMovieMeteringMode(hCamera, lMeteringMode);
    
    return result;
}

/**
 * @brief Gets the metering mode setting in movie mode.
 *
 * @param hCamera Camera handle.
 * @param plMeteringMode Returns the current value for metering mode.
 * @return API result.
 */
-(long) XSDK_GetMovieMeteringMode:(XSDK_HANDLE) hCamera
                   plMeteringMode:(long*) plMeteringMode
{
    long result = pXSDK_GetMovieMeteringMode(hCamera, plMeteringMode);
    
    return result;
}

/**
 * @brief Queries supported white-balance modes to set in movie mode.
 *
 * @param hCamera Camera handle.
 * @param plNumWBMode Returns the number of supported XSDK_SetMovieWBMode settings.
 * @param plWBMode If not NULL, plWBMode returns a list of the XSDK_SetMovieWBMode settings supported.
 * @return API result.
 */
-(long) XSDK_CapMovieWBMode:(XSDK_HANDLE) hCamera
                plNumWBMode:(long*) plNumWBMode
                   plWBMode:(long*) plWBMode
{
    long result = XSDK_ERROR;
    
    if(plWBMode == NULL)
    {
        result = pXSDK_CapMovieWBMode(hCamera, plNumWBMode, NULL);
    }
    else
    {
        long list[*plNumWBMode];
        
        result = pXSDK_CapMovieWBMode(hCamera, plNumWBMode, list);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNumWBMode; i++)
            {
                plWBMode[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Sets the white-balance mode in movie mode.
 *
 * @param hCamera Camera handle.
 * @param lWBMode The white-balance mode.
 * @return API result.
 */
-(long) XSDK_SetMovieWBMode:(XSDK_HANDLE) hCamera
                    lWBMode:(long) lWBMode
{
    long result = pXSDK_SetMovieWBMode(hCamera, lWBMode);
    
    return result;
}

/**
 * @brief Gets the white-balance mode setting in movie mode.
 *
 * @param hCamera Camera handle.
 * @param plWBMode The current white-balance mode.
 * @return API result.
 */
-(long) XSDK_GetMovieWBMode:(XSDK_HANDLE) hCamera
                   plWBMode:(long*) plWBMode
{
    long result = pXSDK_GetMovieWBMode(hCamera, plWBMode);
    
    return result;
}

/**
 * @brief Queries supported color temperatures to set available in movie mode when WBMode = ColorTemperature.
 *
 * @param hCamera Camera handle.
 * @param plNumWBColorTemp Returns the number of supported XSDK_SetMovieWBColorTemp settings.
 * @param plWBColorTemp If not NULL, plWBColorTemp returns a list of the XSDK_SetMovieWBColorTemp settings supported.
 * @return API result.
 */
-(long) XSDK_CapMovieWBColorTemp:(XSDK_HANDLE) hCamera
                plNumWBColorTemp:(long*) plNumWBColorTemp
                   plWBColorTemp:(long*) plWBColorTemp
{
    long result = XSDK_ERROR;
    
    if(plWBColorTemp == NULL)
    {
        result = pXSDK_CapMovieWBColorTemp(hCamera, plNumWBColorTemp, NULL);
    }
    else
    {
        long list[*plNumWBColorTemp];
        
        result = pXSDK_CapMovieWBColorTemp(hCamera, plNumWBColorTemp, list);
        if (result == XSDK_COMPLETE)
        {
            for(int i = 0; i < *plNumWBColorTemp; i++)
            {
                plWBColorTemp[i] = list[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Sets the color temperature value in movie mode for WBMode = ColorTemperature.
 *
 * @param hCamera Camera handle.
 * @param lColorTemp The COLOR TEMPERATURE, in degrees Kelvin
 * @return API result.
 */
-(long) XSDK_SetMovieWBColorTemp:(XSDK_HANDLE) hCamera
                      lColorTemp:(long) lColorTemp
{
    long result = pXSDK_SetMovieWBColorTemp(hCamera, lColorTemp);
    
    return result;
}

/**
 * @brief Gets the movie color temperature setting in movie mode for WBMode = ColorTemperature.
 *
 * @param hCamera Camera handle.
 * @param plColorTemp Returns the current COLOR TEMPERATURE, in degrees Kelvin.
 * @return API result.
 */
-(long) XSDK_GetMovieWBColorTemp:(XSDK_HANDLE) hCamera
                     plColorTemp:(long*) plColorTemp
{
    long result = pXSDK_GetMovieWBColorTemp(hCamera, plColorTemp);
    
    return result;
}

/**
 * @brief Queries supported values for a model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_CapProp_PP:(XSDK_HANDLE) hCamera
            lAPICode:(long) lAPICode
           lAPIParam:(long) lAPIParam
              param1:(void*) param1
              param2:(void*) param2
{
    long result = XSDK_ERROR;
    
    switch (lAPICode)
    {
        case 0x2259:
        {
            long* par1 = (long*)param1;
            if(param2 == NULL)
            {
                result = pXSDK_CapProp(hCamera, lAPICode, lAPIParam, par1, NULL);
                if (result == XSDK_COMPLETE)
                {
                    param1 = par1;
                }
            }
            else
            {
                SDK_FOCUS_POS_CAP** par2 = static_cast<SDK_FOCUS_POS_CAP**>(param2);
                
                result = pXSDK_CapProp(hCamera, lAPICode, lAPIParam, par1, par2);
            }
            break;
        }
        case 0x2287:
        {
            long* par1 = (long*)param1;
            if(param2 == NULL)
            {
                result = pXSDK_CapProp(hCamera, lAPICode, lAPIParam, par1, NULL);
                if (result == XSDK_COMPLETE)
                {
                    param1 = par1;
                }
            }
            else
            {
                SDK_AFZoneCustomCapablity** par2 = static_cast<SDK_AFZoneCustomCapablity**>(param2);
                
                result = pXSDK_CapProp(hCamera, lAPICode, lAPIParam, par1, par2);
            }
            break;
        }
        case 0x362B:
        {
            SDK_MOVIE_AFC_CUSTOM* par1 = static_cast<SDK_MOVIE_AFC_CUSTOM*>(param1);
            SDK_MOVIE_AFC_CUSTOM* par2 = static_cast<SDK_MOVIE_AFC_CUSTOM*>(param2);
            
            result = pXSDK_CapProp(hCamera, lAPICode, lAPIParam, par1, par2);
            
            break;
        }
        case 0x4237:
        case 0x4238:
        {
            long* par1 = (long*)param1;
            long* par2 = (long*)param2;
            result = pXSDK_CapProp(hCamera, lAPICode, lAPIParam, par1, par2);
            if (result == XSDK_COMPLETE)
            {
                param1 = par1;
                param2 = par2;
            }
            
            break;
        }
        default:
        {
            long* par1 = (long*)param1;
            if(param2 == NULL)
            {
                result = pXSDK_CapProp(hCamera, lAPICode, lAPIParam, par1, NULL);
                if (result == XSDK_COMPLETE)
                {
                    param1 = par1;
                }
            }
            else
            {
                long list[*par1];
                long* par2 = (long*)param2;
                result = pXSDK_CapProp(hCamera, lAPICode, lAPIParam, par1, list);
                if (result == XSDK_COMPLETE)
                {
                    param1 = par1;
                    for(int i = 0; i < *par1; i++)
                    {
                        par2[i] = list[i];
                    }
                }
            }
            break;
        }
    }
    
    return result;
}

/**
 * @brief Queries supported values for a model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_CapProp_LPP:(XSDK_HANDLE) hCamera
                lAPICode:(long) lAPICode
               lAPIParam:(long) lAPIParam
                  param1:(long) param1
                  param2:(void*) param2
                  param3:(void*) param3
{
    long result = XSDK_ERROR;
    
    switch (lAPICode)
    {
        case 0x2213:
        case 0x2235:
        {
            SDK_FocusArea* par2 = static_cast<SDK_FocusArea*>(param2);
            SDK_FocusArea* par3 = static_cast<SDK_FocusArea*>(param3);
            
            result = pXSDK_CapProp(hCamera, lAPICode, lAPIParam, param1, par2, par3);
            
            break;
        }
        default:
        {
            long* par2 = (long*)param2;
            if(param3 == NULL)
            {
                result = pXSDK_CapProp(hCamera, lAPICode, lAPIParam, param1, par2, NULL);
                if (result == XSDK_COMPLETE)
                {
                    param2 = par2;
                }
            }
            else
            {
                long list[*par2];
                long* par3 = (long*)param3;
                result = pXSDK_CapProp(hCamera, lAPICode, lAPIParam, param1, par2, list);
                if (result == XSDK_COMPLETE)
                {
                    param2 = par2;
                    for(int i = 0; i < *par2; i++)
                    {
                        par3[i] = list[i];
                    }
                }
            }
            break;
        }
    }
    
    return result;
}

/**
 * @brief Queries supported values for a model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_CapProp_PPPP:(XSDK_HANDLE) hCamera
                 lAPICode:(long) lAPICode
                lAPIParam:(long) lAPIParam
                   param1:(void*) param1
                   param2:(void*) param2
                   param3:(void*) param3
                   param4:(void*) param4
{
    long result = XSDK_ERROR;
    
    switch (lAPICode)
    {
        case 0x2272:
        case 0x2274:
        {
            long* par1 = (long*)param1;
            long* par3 = (long*)param3;
            if(param2 == NULL && param4 == NULL)
            {
                result = pXSDK_CapProp(hCamera, lAPICode, lAPIParam, par1, NULL, par3, NULL);
                if (result == XSDK_COMPLETE)
                {
                    param1 = par1;
                    param3 = par3;
                }
            }
            else
            {
                long list1[*par1];
                long list2[*par3];
                long* par2 = (long*)param2;
                long* par4 = (long*)param4;
                result = pXSDK_CapProp(hCamera, lAPICode, lAPIParam, par1, list1, par3, list2);
                if (result == XSDK_COMPLETE)
                {
                    param1 = par1;
                    for(int i = 0; i < *par1; i++)
                    {
                        par2[i] = list1[i];
                    }
                    param3 = par3;
                    for(int i = 0; i < *par3; i++)
                    {
                        par4[i] = list2[i];
                    }
                }
            }
            break;
        }
        default:
        {
            long* par1 = (long*)param1;
            long* par2 = (long*)param2;
            if(param3 == NULL && param4 == NULL)
            {
                result = pXSDK_CapProp(hCamera, lAPICode, lAPIParam, par1, par2, NULL, NULL);
                if (result == XSDK_COMPLETE)
                {
                    param1 = par1;
                    param2 = par2;
                }
            }
            else
            {
                long list1[*par1];
                long list2[*par2];
                long* par3 = (long*)param3;
                long* par4 = (long*)param4;
                result = pXSDK_CapProp(hCamera, lAPICode, lAPIParam, par1, par2, list1, list2);
                if (result == XSDK_COMPLETE)
                {
                    param1 = par1;
                    for(int i = 0; i < *par1; i++)
                    {
                        par3[i] = list1[i];
                    }
                    param2 = par2;
                    for(int i = 0; i < *par2; i++)
                    {
                        par4[i] = list2[i];
                    }
                }
            }
            break;
        }
    }
    
    return result;
}

/**
 * @brief Queries supported values for a model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_CapProp_LPPPP:(XSDK_HANDLE) hCamera
                  lAPICode:(long) lAPICode
                 lAPIParam:(long) lAPIParam
                    param1:(long) param1
                    param2:(void*) param2
                    param3:(void*) param3
                    param4:(void*) param4
                    param5:(void*) param5
{
    long result = XSDK_ERROR;
    
    long* par2 = (long*)param2;
    long* par3 = (long*)param3;
    long* par4 = (long*)param4;
    long* par5 = (long*)param5;
    result = pXSDK_CapProp(hCamera, lAPICode, lAPIParam, param1, par2, par3, par4, par5);
    if (result == XSDK_COMPLETE)
    {
        param2 = par2;
        param3 = par3;
        param4 = par4;
        param5 = par5;
    }
    
    return result;
}

/**
 * @brief Queries supported values for a model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_CapProp_PPPPPP:(XSDK_HANDLE) hCamera
                   lAPICode:(long) lAPICode
                  lAPIParam:(long) lAPIParam
                     param1:(void*) param1
                     param2:(void*) param2
                     param3:(void*) param3
                     param4:(void*) param4
                     param5:(void*) param5
                     param6:(void*) param6
{
    long result = XSDK_ERROR;
    
    long* par1 = (long*)param1;
    long* par2 = (long*)param2;
    long* par3 = (long*)param3;
    if(param4 == NULL && param5 == NULL && param6 == NULL)
    {
        result = pXSDK_CapProp(hCamera, lAPICode, lAPIParam, par1, par2, par3, NULL, NULL, NULL);
        if (result == XSDK_COMPLETE)
        {
            param1 = par1;
            param2 = par2;
            param3 = par3;
        }
    }
    else
    {
        long list1[*par1];
        long list2[*par2];
        long list3[*par3];
        long* par4 = (long*)param4;
        long* par5 = (long*)param5;
        long* par6 = (long*)param6;
        result = pXSDK_CapProp(hCamera, lAPICode, lAPIParam, par1, par2, par3, list1, list2, list3);
        if (result == XSDK_COMPLETE)
        {
            param1 = par1;
            for(int i = 0; i < *par1; i++)
            {
                par4[i] = list1[i];
            }
            param2 = par2;
            for(int i = 0; i < *par2; i++)
            {
                par5[i] = list2[i];
            }
            param3 = par3;
            for(int i = 0; i < *par3; i++)
            {
                par6[i] = list3[i];
            }
        }
    }
    
    return result;
}

/**
 * @brief Sets values for the model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_SetProp:(XSDK_HANDLE) hCamera
            lAPICode:(long) lAPICode
           lAPIParam:(long) lAPIParam
{
    long result = pXSDK_SetProp(hCamera, lAPICode, lAPIParam);
    
    return result;
}

/**
 * @brief Sets values for the model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_SetProp_L:(XSDK_HANDLE) hCamera
              lAPICode:(long) lAPICode
             lAPIParam:(long) lAPIParam
                param1:(long) param1
{
    long result = pXSDK_SetProp(hCamera, lAPICode, lAPIParam, param1);
    
    return result;
}

/**
 * @brief Sets values for the model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_SetProp_P:(XSDK_HANDLE) hCamera
              lAPICode:(long) lAPICode
             lAPIParam:(long) lAPIParam
                param1:(void*) param1
{
    long result = XSDK_ERROR;
    
    switch (lAPICode)
    {
        case 0x2167:
        {
            SDK_FaceFrameInfo* par1 = static_cast<SDK_FaceFrameInfo*>(param1);
            
            result = pXSDK_SetProp(hCamera, lAPICode, lAPIParam, par1);
            
            break;
        }
        case 0x362C:
        {
            SDK_MOVIE_AFC_CUSTOM* par1 = static_cast<SDK_MOVIE_AFC_CUSTOM*>(param1);
            
            result = pXSDK_SetProp(hCamera, lAPICode, lAPIParam, par1);
            
            break;
        }
        default:
        {
            result = XSDK_ERROR;
            break;
        }
    }
    
    return result;
}

/**
 * @brief Sets values for the model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_SetProp_S:(XSDK_HANDLE) hCamera
              lAPICode:(long) lAPICode
             lAPIParam:(long) lAPIParam
                param1:(NSString*) param1
{
    char* par1 = (char*)[param1 UTF8String];
    
    long result = pXSDK_SetProp(hCamera, lAPICode, lAPIParam, par1);
    
    return result;
}

/**
 * @brief Sets values for the model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_SetProp_LL:(XSDK_HANDLE) hCamera
               lAPICode:(long) lAPICode
              lAPIParam:(long) lAPIParam
                 param1:(long) param1
                 param2:(long) param2
{
    long result = pXSDK_SetProp(hCamera, lAPICode, lAPIParam, param1, param2);
    
    return result;
}

/**
 * @brief Sets values for the model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_SetProp_LP:(XSDK_HANDLE) hCamera
               lAPICode:(long) lAPICode
              lAPIParam:(long) lAPIParam
                 param1:(long) param1
                 param2:(void*) param2
{
    long result = XSDK_ERROR;
    
    switch (lAPICode)
    {
        case 0x2205:
        case 0x2214:
        {
            SDK_FocusArea* par2 = static_cast<SDK_FocusArea*>(param2);
            
            result = pXSDK_SetProp(hCamera, lAPICode, lAPIParam, param1, par2);
            
            break;
        }
        case 0x2288:
        {
            SDK_AFZoneCustomCapablity* par2 = static_cast<SDK_AFZoneCustomCapablity*>(param2);
            
            result = pXSDK_SetProp(hCamera, lAPICode, lAPIParam, param1, par2);
            
            break;
        }
        case 0x2353:
        case 0x2357:
        {
            SDK_CustomWBArea* par2 = static_cast<SDK_CustomWBArea*>(param2);
            
            result = pXSDK_SetProp(hCamera, lAPICode, lAPIParam, param1, par2);
            
            break;
        }
        case 0x4211:
        {
            SDK_FrameGuideGridInfo* par2 = static_cast<SDK_FrameGuideGridInfo*>(param2);
            
            result = pXSDK_SetProp(hCamera, lAPICode, lAPIParam, param1, par2);
            
            break;
        }
        default:
        {
            result = XSDK_ERROR;
            break;
        }
    }
    
    return result;
}

/**
 * @brief Sets values for the model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_SetProp_LS:(XSDK_HANDLE) hCamera
               lAPICode:(long) lAPICode
              lAPIParam:(long) lAPIParam
                 param1:(long) param1
                 param2:(NSString*) param2
{
    char* par2 = (char*)[param2 UTF8String];
    
    long result = pXSDK_SetProp(hCamera, lAPICode, lAPIParam, param1, par2);
    
    return result;
}

/**
 * @brief Sets values for the model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_SetProp_SS:(XSDK_HANDLE) hCamera
               lAPICode:(long) lAPICode
              lAPIParam:(long) lAPIParam
                 param1:(NSString*) param1
                 param2:(NSString*) param2
{
    char* par1 = (char*)[param1 UTF8String];
    char* par2 = (char*)[param2 UTF8String];
    
    long result = pXSDK_SetProp(hCamera, lAPICode, lAPIParam, par1, par2);
    
    return result;
}

/**
 * @brief Sets values for the model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_SetProp_LLL:(XSDK_HANDLE) hCamera
                lAPICode:(long) lAPICode
               lAPIParam:(long) lAPIParam
                  param1:(long) param1
                  param2:(long) param2
                  param3:(long) param3
{
    long result = pXSDK_SetProp(hCamera, lAPICode, lAPIParam, param1, param2, param3);
    
    return result;
}

/**
 * @brief Sets values for the model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_SetProp_LSL:(XSDK_HANDLE) hCamera
                lAPICode:(long) lAPICode
               lAPIParam:(long) lAPIParam
                  param1:(long) param1
                  param2:(NSString*) param2
                  param3:(long) param3
{
    char* par2 = (char*)[param2 UTF8String];
    
    long result = pXSDK_SetProp(hCamera, lAPICode, lAPIParam, param1, par2, param3);
    
    return result;
}

/**
 * @brief Sets values for the model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_SetProp_LLLLL:(XSDK_HANDLE) hCamera
                  lAPICode:(long) lAPICode
                 lAPIParam:(long) lAPIParam
                    param1:(long) param1
                    param2:(long) param2
                    param3:(long) param3
                    param4:(long) param4
                    param5:(long) param5
{
    long result = pXSDK_SetProp(hCamera, lAPICode, lAPIParam, param1, param2, param3, param4, param5);
    
    return result;
}

/**
 * @brief Sets values for the model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_SetProp_LLLLLL:(XSDK_HANDLE) hCamera
                   lAPICode:(long) lAPICode
                  lAPIParam:(long) lAPIParam
                     param1:(long) param1
                     param2:(long) param2
                     param3:(long) param3
                     param4:(long) param4
                     param5:(long) param5
                     param6:(long) param6
{
    long result = pXSDK_SetProp(hCamera, lAPICode, lAPIParam, param1, param2, param3, param4, param5, param6);
    
    return result;
}

/**
 * @brief Gets the settings for the model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_GetProp_P:(XSDK_HANDLE) hCamera
              lAPICode:(long) lAPICode
             lAPIParam:(long) lAPIParam
                param1:(void*) param1
{
    long result = XSDK_ERROR;
    
    switch (lAPICode)
    {
        case 0x2269:
        {
            SDK_CropAreaFrameInfo* par1 = static_cast<SDK_CropAreaFrameInfo*>(param1);
            
            result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1);
            
            break;
        }
        case 0x226B:
        {
            SDK_FOCUS_LIMITER_INDICATOR* par1 = static_cast<SDK_FOCUS_LIMITER_INDICATOR*>(param1);
            
            result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1);
            
            break;
        }
        case 0x228A:
        {
            SDK_TrackingAfFrameInfo* par1 = static_cast<SDK_TrackingAfFrameInfo*>(param1);
            
            result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1);
            
            break;
        }
        case 0x357B:
        {
            SDK_MovieTransparentFrameInfo* par1 = static_cast<SDK_MovieTransparentFrameInfo*>(param1);
            
            result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1);
            
            break;
        }
        case 0x362D:
        {
            SDK_MOVIE_AFC_CUSTOM* par1 = static_cast<SDK_MOVIE_AFC_CUSTOM*>(param1);
            
            result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1);
            
            break;
        }
        case 0x3922:
        {
            SDK_MICLEVEL_INDICATOR* par1 = static_cast<SDK_MICLEVEL_INDICATOR*>(param1);
            
            result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1);
            
            break;
        }
        case 0x4044:
        case 0x407E:
        {
            char* par1 = static_cast<char*>(param1);
            
            result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1);
            
            break;
        }
        default:
        {
            long* par1 = (long*)param1;
            
            result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1);
            
            break;
        }
    }
    
    return result;
}

/**
 * @brief Gets the settings for the model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_GetProp_PP:(XSDK_HANDLE) hCamera
               lAPICode:(long) lAPICode
              lAPIParam:(long) lAPIParam
                 param1:(void*) param1
                 param2:(void*) param2
{
    long result = XSDK_ERROR;
    
    switch (lAPICode)
    {
        case 0x2166:
        {
            long* par1 = (long*)param1;
            if(param2 == NULL)
            {
                result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1, NULL);
                if (result == XSDK_COMPLETE)
                {
                    param1 = par1;
                }
            }
            else
            {
                SDK_FaceFrameInfo** par2 = static_cast<SDK_FaceFrameInfo**>(param2);
                
                result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1, par2);
                if (result == XSDK_COMPLETE)
                {
                    param1 = par1;
                }
            }
            
            break;
        }
        case 0x226C:
        {
            long* par1 = (long*)param1;
            if(param2 == NULL)
            {
                result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1, NULL);
                if (result == XSDK_COMPLETE)
                {
                    param1 = par1;
                }
            }
            else
            {
                SDK_FOCUS_LIMITER** par2 = static_cast<SDK_FOCUS_LIMITER**>(param2);
                
                result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1, par2);
            }
            
            break;
        }
        case 0x3555:
        {
            long* par1 = (long*)param1;
            if(param2 == NULL)
            {
                result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1, NULL);
                if (result == XSDK_COMPLETE)
                {
                    param1 = par1;
                }
            }
            else
            {
                SDK_HistogramData** par2 = static_cast<SDK_HistogramData**>(param2);
                
                result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1, par2);
            }
            
            break;
        }
        case 0x3556:
        {
            long* par1 = (long*)param1;
            if(param2 == NULL)
            {
                result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1, NULL);
                if (result == XSDK_COMPLETE)
                {
                    param1 = par1;
                }
            }
            else
            {
                SDK_FocusMapData** par2 = static_cast<SDK_FocusMapData**>(param2);
                
                result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1, par2);
            }
            
            break;
        }
        case 0x355A:
        case 0x355B:
        case 0x355C:
        {
            long* par1 = (long*)param1;
            uint8_t* par2 = (uint8_t*)param2;
            
            result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1, par2);
            
            break;
        }
        case 0x4046:
        {
            char* par1 = static_cast<char*>(param1);
            char* par2 = static_cast<char*>(param2);
            
            result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1, par2);
            
            break;
        }
        default:
        {
            long* par1 = (long*)param1;
            long* par2 = (long*)param2;
            
            result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1, par2);
            
            break;
        }
    }
    
    return result;
}

/**
 * @brief Gets the settings for the model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_GetProp_LP:(XSDK_HANDLE) hCamera
               lAPICode:(long) lAPICode
              lAPIParam:(long) lAPIParam
                 param1:(long) param1
                 param2:(void*) param2
{
    long result = XSDK_ERROR;
    
    switch (lAPICode)
    {
        case 0x2206:
        case 0x2215:
        {
            SDK_FocusArea* par2 = static_cast<SDK_FocusArea*>(param2);
            
            result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, param1, par2);
            
            break;
        }
        case 0x2289:
        {
            SDK_AFZoneCustomCapablity* par2 = static_cast<SDK_AFZoneCustomCapablity*>(param2);
            
            result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, param1, par2);
            
            break;
        }
        case 0x2354:
        case 0x2358:
        {
            SDK_CustomWBArea* par2 = static_cast<SDK_CustomWBArea*>(param2);
            
            result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, param1, par2);
            
            break;
        }
        case 0x4048:
        {
            char* par2 = static_cast<char*>(param2);
            
            result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, param1, par2);
            
            break;
        }
        case 0x4212:
        {
            SDK_FrameGuideGridInfo* par2 = static_cast<SDK_FrameGuideGridInfo*>(param2);
            
            result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, param1, par2);
            
            break;
        }
        case 0x427A:
        {
            SDK_PlayBackObjectInformation* par2 = static_cast<SDK_PlayBackObjectInformation*>(param2);
            
            result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, param1, par2);
            
            break;
        }
        default:
        {
            long* par2 = (long*)param2;
            
            result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, param1, par2);
            
            break;
        }
    }
    
    return result;
}

/**
 * @brief Gets the settings for the model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_GetProp_PPP:(XSDK_HANDLE) hCamera
                lAPICode:(long) lAPICode
               lAPIParam:(long) lAPIParam
                  param1:(void*) param1
                  param2:(void*) param2
                  param3:(void*) param3
{
    long* par1 = (long*)param1;
    long* par2 = (long*)param2;
    long* par3 = (long*)param3;
    
    long result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1, par2, par3);
    
    return result;
}

/**
 * @brief Gets the settings for the model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_GetProp_LPP:(XSDK_HANDLE) hCamera
                lAPICode:(long) lAPICode
               lAPIParam:(long) lAPIParam
                  param1:(long) param1
                  param2:(void*) param2
                  param3:(void*) param3
{
    long result = XSDK_ERROR;
    
    switch (lAPICode)
    {
        case 0x4075:
        {
            char* par2 = static_cast<char*>(param2);
            long* par3 = (long*)param3;
            
            result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, param1, par2, par3);
            
            break;
        }
        case 0x4279:
        {
            long* par2 = (long*)param2;
            if(param3 == NULL)
            {
                result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, param1, par2, NULL);
                if (result == XSDK_COMPLETE)
                {
                    param2 = par2;
                }
            }
            else
            {
                long list[*par2];
                long* par3 = (long*)param3;
                result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, param1, par2, list);
                if (result == XSDK_COMPLETE)
                {
                    param2 = par2;
                    for(int i = 0; i < *par2; i++)
                    {
                        par3[i] = list[i];
                    }
                }
            }
            
            break;
        }
        default:
        {
            long* par2 = (long*)param2;
            long* par3 = (long*)param3;
            
            result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, param1, par2, par3);
            
            break;
        }
    }
    
    return result;
}

/**
 * @brief Gets the settings for the model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_GetProp_LLP:(XSDK_HANDLE) hCamera
                lAPICode:(long) lAPICode
               lAPIParam:(long) lAPIParam
                  param1:(long) param1
                  param2:(long) param2
                  param3:(void*) param3
{
    uint8_t* par3 = (uint8_t*)param3;
    
    long result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, param1, param2, par3);
    
    return result;
}

/**
 * @brief Gets the settings for the model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_GetProp_PPPP:(XSDK_HANDLE) hCamera
                 lAPICode:(long) lAPICode
                lAPIParam:(long) lAPIParam
                   param1:(void*) param1
                   param2:(void*) param2
                   param3:(void*) param3
                   param4:(void*) param4
{
    long* par1 = (long*)param1;
    long* par2 = (long*)param2;
    long* par3 = (long*)param3;
    long* par4 = (long*)param4;
    
    long result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1, par2, par3, par4);
    
    return result;
}

/**
 * @brief Gets the settings for the model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_GetProp_PPLP:(XSDK_HANDLE) hCamera
                 lAPICode:(long) lAPICode
                lAPIParam:(long) lAPIParam
                   param1:(void*) param1
                   param2:(void*) param2
                   param3:(long) param3
                   param4:(void*) param4
{
    long result = XSDK_ERROR;
    
    long* par1 = (long*)param1;
    long* par2 = (long*)param2;
    if(param4 == NULL)
    {
        result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1, par2, param3, NULL);
        
        if (result == XSDK_COMPLETE)
        {
            param1 = par1;
            param2 = par2;
        }
    }
    else
    {
        SDK_FOLDER_INFO** par4 = static_cast<SDK_FOLDER_INFO**>(param4);
        
        result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1, par2, param3, par4);
        if (result == XSDK_COMPLETE)
        {
            param1 = par1;
            param2 = par2;
        }
    }
    
    return result;
}

/**
 * @brief Gets the settings for the model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_GetProp_LPPPP:(XSDK_HANDLE) hCamera
                  lAPICode:(long) lAPICode
                 lAPIParam:(long) lAPIParam
                    param1:(long) param1
                    param2:(void*) param2
                    param3:(void*) param3
                    param4:(void*) param4
                    param5:(void*) param5
{
    long* par2 = (long*)param2;
    long* par3 = (long*)param3;
    long* par4 = (long*)param4;
    long* par5 = (long*)param5;
    
    long result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, param1, par2, par3, par4, par5);
    
    return result;
}

/**
 * @brief Gets the settings for the model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_GetProp_LLLPP:(XSDK_HANDLE) hCamera
                  lAPICode:(long) lAPICode
                 lAPIParam:(long) lAPIParam
                    param1:(long) param1
                    param2:(long) param2
                    param3:(long) param3
                    param4:(void*) param4
                    param5:(void*) param5
{
    long* par4 = (long*)param4;
    uint8_t* par5 = (uint8_t*)param5;
    
    long result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, param1, param2, param3, par4, par5);
    
    return result;
}

/**
 * @brief Gets the settings for the model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_GetProp_PPPPPP:(XSDK_HANDLE) hCamera
                   lAPICode:(long) lAPICode
                  lAPIParam:(long) lAPIParam
                     param1:(void*) param1
                     param2:(void*) param2
                     param3:(void*) param3
                     param4:(void*) param4
                     param5:(void*) param5
                     param6:(void*) param6
{
    long* par1 = (long*)param1;
    long* par2 = (long*)param2;
    long* par3 = (long*)param3;
    long* par4 = (long*)param4;
    long* par5 = (long*)param5;
    long* par6 = (long*)param6;
    
    long result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1, par2, par3, par4, par5, par6);
    
    return result;
}

/**
 * @brief Gets the settings for the model-dependent function.
 *
 * @return API result.
 */
-(long) XSDK_GetProp_PPPPPPPP:(XSDK_HANDLE) hCamera
                     lAPICode:(long) lAPICode
                    lAPIParam:(long) lAPIParam
                       param1:(void*) param1
                       param2:(void*) param2
                       param3:(void*) param3
                       param4:(void*) param4
                       param5:(void*) param5
                       param6:(void*) param6
                       param7:(void*) param7
                       param8:(void*) param8
{
    long* par1 = (long*)param1;
    long* par2 = (long*)param2;
    long* par3 = (long*)param3;
    long* par4 = (long*)param4;
    long* par5 = (long*)param5;
    long* par6 = (long*)param6;
    long* par7 = (long*)param7;
    long* par8 = (long*)param8;
    
    long result = pXSDK_GetProp(hCamera, lAPICode, lAPIParam, par1, par2, par3, par4, par5, par6, par7, par8);
    
    return result;
}

@end
