//
//  ContentView.swift
//  MultipleCameraControl
//
//  Created by saito shota on 2024/10/04.
//

import SwiftUI

struct ContentView: View {
    @State private var isUSBTether = true
    @State private var isIPTether = false
    @State private var sdkAlert = false
    @State private var lodaAlert = false
    @State private var cameraIndex = 0
    @State private var cameraCount = 0
    @State private var indexList: [Int] = []
    @State private var detectedList: [DetectedListInfo] = []
    @State private var connectedList: [ConnectedListInfo] = []
    @State private var isDetectEnabled = true
    @State private var isAppendEnabled = false
    @State private var isOpenEnabled = false
    @State private var isCloseEnabled = false
    @State private var isGetInfoEnabled = false
    @State private var cameraControl : CameraControl? = CameraControl()
    
    var body: some View {
        VStack {
            GroupBox(label: Label("Detect / Append", systemImage:"")){
                HStack{
                    GroupBox(label: Label("Interface", systemImage:"")){
                        HStack{
                            Toggle(isOn: $isUSBTether){
                                Label("USB", systemImage:"")
                            }
                            Toggle(isOn: $isIPTether){
                                Label("IP(Local network)", systemImage:"")
                            }
                        }
                    }
                    VStack{
                        HStack{
                            Spacer()
                            Button(action: {
                                var interface : Int = 0
                                if(isUSBTether)
                                {
                                    interface |= Int(XSDK_DSC_IF_USB)
                                }
                                if(isIPTether)
                                {
                                    interface |= Int(XSDK_DSC_IF_WIFI_LOCAL)
                                }
                                indexList = []
                                cameraCount = 0
                                detectedList = []
                                
                                let result : Int = cameraControl!.Detect(interface: interface, ipAddess: "", deviceName: "", count: &cameraCount) // Camera search.
                                if(result != XSDK_COMPLETE)
                                {
                                    // error.
                                    sdkAlert = true
                                }
                                else
                                {
                                    for i in 0..<cameraCount // Update list.
                                    {
                                        let info = DetectedListInfo(DeviceIndex: String(i), Valid: "Valid", Product: "", SerialNoIP: "")
                                        detectedList.append(info)
                                        indexList.append(i)
                                    }
                                }
                                
                                if(detectedList.count > 0) // Button state setting.
                                {
                                    isAppendEnabled = true
                                    isOpenEnabled = true
                                }
                                else
                                {
                                    isAppendEnabled = false
                                    isOpenEnabled = false
                                    isCloseEnabled = false
                                    isGetInfoEnabled = false
                                }
                            }, label: {
                                    Text("Detect")
                                    .frame(width: 100, height: 20)
                            })
                            .disabled(!isDetectEnabled)
                            Button(action: {
                                var interface : Int = 0
                                if(isUSBTether)
                                {
                                    interface |= Int(XSDK_DSC_IF_USB)
                                }
                                if(isIPTether)
                                {
                                    interface |= Int(XSDK_DSC_IF_WIFI_LOCAL)
                                }
                                indexList = []
                                detectedList = []
                                cameraCount = 128
                                var cameraList = Array(repeating: CameraList(), count: cameraCount)
                                for i in 0..<cameraCount
                                {
                                    cameraList[i] = CameraList()
                                }
                                
                                let result : Int = cameraControl!.Append(interface: interface, ipAddess: "", deviceName: "", count: &cameraCount, cameraList: &cameraList) // Camera list acquisition.
                                if(result != XSDK_COMPLETE)
                                {
                                    sdkAlert = true
                                }
                                else
                                {
                                    for i in 0..<cameraCount // Update list.
                                    {
                                        var valid = "Valid"
                                        if(!cameraList[i].bValid)
                                        {
                                            valid = "Invalid"
                                        }
                                        var serialNoIP = cameraList[i].strSerialNo
                                        if(cameraList[i].strFramework != "USB")
                                        {
                                            serialNoIP = cameraList[i].strIPAddress
                                        }
                                        let info = DetectedListInfo(DeviceIndex: String(i), Valid: valid, Product: cameraList[i].strProduct, SerialNoIP: serialNoIP!)
                                        detectedList.append(info)
                                        indexList.append(i)
                                    }
                                    
                                    isOpenEnabled = true
                                    if(detectedList[cameraIndex].Valid == "Invalid") // Button state setting.
                                    {
                                        isOpenEnabled = false
                                    }
                                }
                            }, label: {
                                Text("Append")
                                .frame(width: 100, height: 20)
                            })
                            .disabled(!isAppendEnabled)
                        }
                        HStack{
                            Spacer()
                            Text("# of cameras:")
                            Text(String(cameraCount))
                        }
                    }
                }
                HStack{
                    Text("Detected cameras")
                    Spacer()
                }
                Table(detectedList){
                    TableColumn("      index"){ item in
                        Text(item.DeviceIndex)
                            .frame(maxWidth: .infinity, alignment: .center)
                            .multilineTextAlignment(.center)
                            .padding(0)
                    }.width(min: 70, max: 70)
                    TableColumn("            bValid"){ item in
                        Text(item.Valid)
                            .frame(maxWidth: .infinity, alignment: .center)
                            .multilineTextAlignment(.center)
                            .padding(0)
                    }.width(min: 110, max: 110)
                    TableColumn("        strProduct"){ item in
                        Text(item.Product)
                            .frame(maxWidth: .infinity, alignment: .center)
                            .multilineTextAlignment(.center)
                            .padding(0)
                    }.width(min: 110, max: 110)
                    TableColumn("                       strSerialNo/strIPAddress"){ item in
                        Text(item.SerialNoIP)
                            .frame(maxWidth: .infinity, alignment: .center)
                            .multilineTextAlignment(.center)
                            .padding(0)
                    }.width(min: 280, max: 280)
                }
            }
            .padding(.horizontal, 0)
            .frame(minWidth: 670, maxWidth: 670, minHeight: 330, maxHeight: 330)
            GroupBox(label: Label("Individual camera control", systemImage:"")){
                HStack{
                    Picker("index:", selection: $cameraIndex){
                        ForEach(indexList, id: \.self) { fruit in
                            Text(String(fruit))
                            .tag(fruit)
                        }
                    }
                    .frame(width: 150, height: 20)
                    .onChange(of: cameraIndex){ value in
                        if(connectedList.first(where: {$0.DeviceIndex == value}) == nil) // Button state setting.
                        {
                            isOpenEnabled = true
                            isCloseEnabled = false
                            isGetInfoEnabled = false
                        }
                        else
                        {
                            isOpenEnabled = false
                            isCloseEnabled = true
                            isGetInfoEnabled = true
                        }
                        if(detectedList[value].Valid == "Invalid")
                        {
                            isOpenEnabled = false
                        }
                    }
                    Button(action: {
                        var hCamera : XSDK_HANDLE? = UnsafeMutablePointer<UnsafeMutablePointer<Int8>?>.allocate(capacity: 1)
                        var camaraMode : Int = 0
                        let option : UnsafeMutableRawPointer? = nil
                        let device : String = "ENUM:" + String(cameraIndex)
                        
                        var result : Int = cameraControl!.Open(device: device, hCamera: &hCamera, cameraMode: &camaraMode, optino: option) // Establish a session between the camera.
                        if(result != XSDK_COMPLETE)
                        {
                            // error.
                            sdkAlert = true
                        }
                        else
                        {
                            var info : DeviceInformation = DeviceInformation()
                            result = cameraControl!.GetDeviceInfo(hCamera: hCamera!, deviceInfo: &info) // Get information about the connect camera.
                            if(result != XSDK_COMPLETE)
                            {
                                // error.
                                result = cameraControl!.Close(hCamera: hCamera!)
                                sdkAlert = true
                            }
                            else
                            {
                                // Update list.
                                let info = ConnectedListInfo(DeviceIndex: cameraIndex, Vendor: info.strVendor, Product: info.strProduct, Firmware: info.strFirmware, Sensitivity: "", hCamera: hCamera!)
                                connectedList.append(info)
                                connectedList.sort { $0.DeviceIndex < $1.DeviceIndex }
                            }
                        }
                        
                        if(connectedList.count > 0) // Button state setting.
                        {
                            isDetectEnabled = false
                        }
                        if(connectedList.first(where: { $0.DeviceIndex == cameraIndex }) != nil)
                        {
                            isOpenEnabled = false
                            isCloseEnabled = true
                            isGetInfoEnabled = true
                        }
                    }){
                        Text("Open")
                        .frame(width: 100, height: 20)
                    }
                    .disabled(!isOpenEnabled)
                    Button(action: {
                        if let camera = connectedList.first(where: { $0.DeviceIndex == cameraIndex })
                        {
                            let result : Int = cameraControl!.Close(hCamera: camera.hCamera) // Disestablish a session between the camera.
                            if(result != XSDK_COMPLETE)
                            {
                                // error.
                                sdkAlert = true
                            }
                            connectedList.removeAll { $0.DeviceIndex == cameraIndex } // Update list.
                        }
                        
                        if(connectedList.first(where: { $0.DeviceIndex == cameraIndex }) == nil) // Button state setting.
                        {
                            isOpenEnabled = true
                            isCloseEnabled = false
                            isGetInfoEnabled = false
                        }
                        if(connectedList.count == 0)
                        {
                            isDetectEnabled = true
                        }
                    }){
                        Text("Close")
                        .frame(width: 100, height: 20)
                    }
                    .disabled(!isCloseEnabled)
                    Spacer()
                }
                HStack{
                    Text("Camera control API:")
                    Spacer()
                }
                Button(action: {
                    if var camera = connectedList.first(where: { $0.DeviceIndex == cameraIndex })
                    {
                        var sensitivity : Int = 0
                        
                        let result : Int = cameraControl!.GetSensitivity(hCamera: camera.hCamera, sensitivity: &sensitivity) // Get the ISO sensitivity.
                        if(result != XSDK_COMPLETE)
                        {
                            // error.
                            sdkAlert = true
                        }
                        else
                        {
                            // Update list.
                            connectedList.removeAll { $0.DeviceIndex == cameraIndex }
                            if let strSensitivity = sensitivityToString[sensitivity] {
                                camera.Sensitivity = strSensitivity
                            } else {
                                let hexString = String(format: "%08x", sensitivity)
                                camera.Sensitivity = "0x" + hexString
                            }
                            connectedList.append(camera)
                            connectedList.sort { $0.DeviceIndex < $1.DeviceIndex }
                        }
                    }
                }){
                    Text("GetSensitivity")
                    .frame(width: 600, height: 20)
                }
                .disabled(!isGetInfoEnabled)
                HStack{
                    Text("Opened devices:")
                    Spacer()
                }
                Table(connectedList){
                    TableColumn("      index"){ item in
                        Text(String(item.DeviceIndex))
                            .frame(maxWidth: .infinity)
                            .multilineTextAlignment(.center)
                            .padding(0)
                    }.width(min: 70, max: 70)
                    TableColumn("          strVendor"){ item in
                        Text(item.Vendor)
                            .frame(maxWidth: .infinity)
                            .multilineTextAlignment(.center)
                            .padding(0)
                    }.width(min: 120, max: 120)
                    TableColumn("          strProduct"){ item in
                        Text(item.Product)
                            .frame(maxWidth: .infinity)
                            .multilineTextAlignment(.center)
                            .padding(0)
                    }.width(min: 120, max: 120)
                    TableColumn("         strFirmware"){ item in
                        Text(item.Firmware)
                            .frame(maxWidth: .infinity)
                            .multilineTextAlignment(.center)
                            .padding(0)
                    }.width(min: 120, max: 120)
                    TableColumn("         Sensitivity"){ item in
                        Text(item.Sensitivity)
                            .frame(maxWidth: .infinity)
                            .multilineTextAlignment(.center)
                            .padding(0)
                    }.width(min: 120, max: 120)
                }
            }
            .padding(.horizontal, 0)
            .frame(minWidth: 670, maxWidth: 670, minHeight: 350, maxHeight: 350)
        }
        .padding(.horizontal, 0)
        .frame(minWidth: 670, maxWidth: 670, minHeight: 700, maxHeight: .infinity)
        .onAppear {
            if (cameraControl!.LoadLibrary()) // Loda XAPI.bundle.
            {
                let result : Int = cameraControl!.Init() // Initialize SDK.
                if(result != XSDK_COMPLETE)
                {
                    // error
                    sdkAlert = true
                }
            }
            else
            {
                // error
                lodaAlert = true
            }
        }
        .alert(isPresented: Binding<Bool>(
            get: { sdkAlert || lodaAlert },
            set: { if !$0 { sdkAlert = false; lodaAlert = false } }
        )) {
            if(lodaAlert)
            {
                Alert(
                    title: Text("Failed to load XSDK.bundle."),
                    message: Text(""),
                    dismissButton: .default(Text("OK")) {
                        exit(0)
                    }
                )
            }
            else
            {
                Alert(
                    title: Text("GetErrNumber()"),
                    message: Text("APICode: \(cameraControl!.strAPICode)\nERRCode: \(cameraControl!.strERRCode)"),
                    dismissButton: .default(Text("OK"))
                )
            }
        }.onDisappear(perform:{
            for i in 0..<connectedList.count
            {
                _ = cameraControl!.Close(hCamera: connectedList[i].hCamera) // Disestablish a session between the camera.
            }
            _ = cameraControl!.Exit() // Termination SDK.
            cameraControl = nil
            exit(0)
        })
    }
}

#Preview {
    ContentView()
}
