//
//  CameraControl.swift
//  MultipleCameraControl
//
//  Created by saito shota on 2024/10/07.
//

import Foundation

let sensitivityToString: [Int: String] =
    [Int(XSDK_SENSITIVITY_ISO40):     "ISO40",
     Int(XSDK_SENSITIVITY_ISO50):     "ISO50",
     Int(XSDK_SENSITIVITY_ISO60):     "ISO60",
     Int(XSDK_SENSITIVITY_ISO64):     "ISO64",
     Int(XSDK_SENSITIVITY_ISO80):     "ISO80",
     Int(XSDK_SENSITIVITY_ISO100):    "ISO100",
     Int(XSDK_SENSITIVITY_ISO125):    "ISO125",
     Int(XSDK_SENSITIVITY_ISO160):    "ISO160",
     Int(XSDK_SENSITIVITY_ISO200):    "ISO200",
     Int(XSDK_SENSITIVITY_ISO250):    "ISO250",
     Int(XSDK_SENSITIVITY_ISO320):    "ISO320",
     Int(XSDK_SENSITIVITY_ISO400):    "ISO400",
     Int(XSDK_SENSITIVITY_ISO500):    "ISO500",
     Int(XSDK_SENSITIVITY_ISO640):    "ISO640",
     Int(XSDK_SENSITIVITY_ISO800):    "ISO800",
     Int(XSDK_SENSITIVITY_ISO1000):   "ISO1000",
     Int(XSDK_SENSITIVITY_ISO1250):   "ISO1250",
     Int(XSDK_SENSITIVITY_ISO1600):   "ISO1600",
     Int(XSDK_SENSITIVITY_ISO2000):   "ISO2000",
     Int(XSDK_SENSITIVITY_ISO2500):   "ISO2500",
     Int(XSDK_SENSITIVITY_ISO3200):   "ISO3200",
     Int(XSDK_SENSITIVITY_ISO4000):   "ISO4000",
     Int(XSDK_SENSITIVITY_ISO5000):   "ISO5000",
     Int(XSDK_SENSITIVITY_ISO6400):   "ISO6400",
     Int(XSDK_SENSITIVITY_ISO8000):   "ISO8000",
     Int(XSDK_SENSITIVITY_ISO10000):  "ISO10000",
     Int(XSDK_SENSITIVITY_ISO12800):  "ISO12800",
     Int(XSDK_SENSITIVITY_ISO16000):  "ISO16000",
     Int(XSDK_SENSITIVITY_ISO20000):  "ISO20000",
     Int(XSDK_SENSITIVITY_ISO25600):  "ISO25600",
     Int(XSDK_SENSITIVITY_ISO32000):  "ISO32000",
     Int(XSDK_SENSITIVITY_ISO40000):  "ISO40000",
     Int(XSDK_SENSITIVITY_ISO51200):  "ISO51200",
     Int(XSDK_SENSITIVITY_ISO64000):  "ISO64000",
     Int(XSDK_SENSITIVITY_ISO80000):  "ISO80000",
     Int(XSDK_SENSITIVITY_ISO102400): "ISO102400",
     Int(XSDK_SENSITIVITY_AUTO_1):    "AUTO_1",
     Int(XSDK_SENSITIVITY_AUTO_2):    "AUTO_2",
     Int(XSDK_SENSITIVITY_AUTO_3):    "AUTO_3",
     Int(XSDK_SENSITIVITY_AUTO_4):    "AUTO_4",
     Int(XSDK_SENSITIVITY_AUTO):      "AUTO",
     Int(XSDK_SENSITIVITY_AUTO400):   "AUTO400",
     Int(XSDK_SENSITIVITY_AUTO800):   "AUTO800",
     Int(XSDK_SENSITIVITY_AUTO1600):  "AUTO1600",
     Int(XSDK_SENSITIVITY_AUTO3200):  "AUTO3200",
     Int(XSDK_SENSITIVITY_AUTO6400):  "AUTO6400"]

let apiCodeToString: [Int: String] =
    [Int(API_CODE_Init):           "API_CODE_Init",
     Int(API_CODE_Exit):           "API_CODE_Exit",
     Int(API_CODE_Detect):         "API_CODE_Detect",
     Int(API_CODE_Append):         "API_CODE_Append",
     Int(API_CODE_Close):          "API_CODE_Close",
     Int(API_CODE_OpenEx):         "API_CODE_OpenEx",
     Int(API_CODE_GetErrorNumber): "API_CODE_GetErrorNumber",
     Int(API_CODE_GetDeviceInfo):  "API_CODE_GetDeviceInfo",
     Int(API_CODE_GetSensitivity): "API_CODE_GetSensitivity",]

let errCodeToString: [Int: String] =
    [Int(XSDK_ERRCODE_NOERR):     "XSDK_ERRCODE_NOERR",
     Int(XSDK_ERRCODE_SEQUENCE):     "XSDK_ERRCODE_SEQUENCE",
     Int(XSDK_ERRCODE_PARAM):     "XSDK_ERRCODE_PARAM",
     Int(XSDK_ERRCODE_INVALID_CAMERA):     "XSDK_ERRCODE_INVALID_CAMERA",
     Int(XSDK_ERRCODE_LOADLIB):     "XSDK_ERRCODE_LOADLIB",
     Int(XSDK_ERRCODE_UNSUPPORTED):    "XSDK_ERRCODE_UNSUPPORTED",
     Int(XSDK_ERRCODE_BUSY):    "XSDK_ERRCODE_BUSY",
     Int(XSDK_ERRCODE_AF_TIMEOUT):    "XSDK_ERRCODE_AF_TIMEOUT",
     Int(XSDK_ERRCODE_SHOOT_ERROR):    "XSDK_ERRCODE_SHOOT_ERROR",
     Int(XSDK_ERRCODE_FRAME_FULL):    "XSDK_ERRCODE_FRAME_FULL",
     Int(XSDK_ERRCODE_STANDBY):    "XSDK_ERRCODE_STANDBY",
     Int(XSDK_ERRCODE_NODRIVER):    "XSDK_ERRCODE_NODRIVER",
     Int(XSDK_ERRCODE_NO_MODEL_MODULE):    "XSDK_ERRCODE_NO_MODEL_MODULE",
     Int(XSDK_ERRCODE_API_NOTFOUND):    "XSDK_ERRCODE_API_NOTFOUND",
     Int(XSDK_ERRCODE_API_MISMATCH):    "XSDK_ERRCODE_API_MISMATCH",
     Int(XSDK_ERRCODE_INVALID_USBMODE):   "XSDK_ERRCODE_INVALID_USBMODE",
     Int(XSDK_ERRCODE_FORCEMODE_BUSY):   "XSDK_ERRCODE_FORCEMODE_BUSY",
     Int(XSDK_ERRCODE_RUNNING_OTHER_FUNCTION):   "XSDK_ERRCODE_RUNNING_OTHER_FUNCTION",
     Int(XSDK_ERRCODE_COMMUNICATION):   "XSDK_ERRCODE_COMMUNICATION",
     Int(XSDK_ERRCODE_TIMEOUT):   "XSDK_ERRCODE_TIMEOUT",
     Int(XSDK_ERRCODE_COMBINATION):   "XSDK_ERRCODE_COMBINATION",
     Int(XSDK_ERRCODE_WRITEERROR):   "XSDK_ERRCODE_WRITEERROR",
     Int(XSDK_ERRCODE_CARDFULL):   "XSDK_ERRCODE_CARDFULL",
     Int(XSDK_ERRCODE_HARDWARE):   "XSDK_ERRCODE_HARDWARE",
     Int(XSDK_ERRCODE_INTERNAL):   "XSDK_ERRCODE_INTERNAL",
     Int(XSDK_ERRCODE_MEMFULL):   "XSDK_ERRCODE_MEMFULL",
     Int(XSDK_ERRCODE_UNKNOWN):   "XSDK_ERRCODE_UNKNOWN"]

class CameraControl
{
    
    var fujiSDK : FujiSDK?
    var apiCode : Int
    var errCode : Int
    var strAPICode : String
    var strERRCode : String
    private var cameraControlExclusion : NSLock
    
    init()
    {
        fujiSDK = FujiSDK()
        apiCode = 0
        errCode = 0
        strAPICode = ""
        strERRCode = ""
        cameraControlExclusion = NSLock()
    }
        
    deinit
    {
        fujiSDK = nil
    }
    
    func CodeNumberToString()
    {
        if let str = apiCodeToString[apiCode] {
            strAPICode = str
        } else {
            let hexString = String(format: "%04x", apiCode)
            strAPICode = "0x" + hexString
        }
        
        if let str = errCodeToString[errCode] {
            strERRCode = str
        } else {
            let hexString = String(format: "%08x", errCode)
            strERRCode = "0x" + hexString
        }
    }
    
    func LoadLibrary() -> Bool
    {
        let result: Bool = fujiSDK!.loadLibrary()
        
        return result
    }
    
    func Init() -> Int
    {
        cameraControlExclusion.lock()
        let result : Int = fujiSDK!.xsdk_Init()
        if(result != XSDK_COMPLETE)
        {
            fujiSDK!.xsdk_GetErrorNumber(nil, plAPICode: &apiCode, plERRCode: &errCode)
            CodeNumberToString()
        }
        
        defer
        {
            cameraControlExclusion.unlock()
        }
        return result
    }
    
    func Exit() -> Int
    {
        cameraControlExclusion.lock()
        let result : Int = fujiSDK!.xsdk_Exit()
        if(result != XSDK_COMPLETE)
        {
            fujiSDK!.xsdk_GetErrorNumber(nil, plAPICode: &apiCode, plERRCode: &errCode)
            CodeNumberToString()
        }
        
        defer
        {
            cameraControlExclusion.unlock()
        }
        return result
    }
    
    func Detect(interface: Int, ipAddess: String, deviceName: String, count: inout Int) -> Int
    {
        cameraControlExclusion.lock()
        let result = fujiSDK!.xsdk_Detect(interface, pInterface: ipAddess, pDeviceName: deviceName, plCount: &count)
        if(result != XSDK_COMPLETE)
        {
            fujiSDK!.xsdk_GetErrorNumber(nil, plAPICode: &apiCode, plERRCode: &errCode)
            CodeNumberToString()
        }
        
        defer
        {
            cameraControlExclusion.unlock()
        }
        return result
    }
    
    func Append(interface: Int, ipAddess: String, deviceName: String, count: inout Int, cameraList: inout [CameraList]) -> Int
    {
        cameraControlExclusion.lock()
        let result = fujiSDK!.xsdk_Append(interface, pInterface: "", pDeviceName: "", plCount: &count, pCameraList: cameraList)
        if(result != XSDK_COMPLETE)
        {
            fujiSDK!.xsdk_GetErrorNumber(nil, plAPICode: &apiCode, plERRCode: &errCode)
            CodeNumberToString()
        }
        
        defer
        {
            cameraControlExclusion.unlock()
        }
        return result
    }
    
    
    func Open(device: String, hCamera: inout XSDK_HANDLE?, cameraMode: inout Int, optino: UnsafeMutableRawPointer?) -> Int
    {
        cameraControlExclusion.lock()
        let result : Int = fujiSDK!.xsdk_OpenEx(device, phCamera: &hCamera, plCameraMode: &cameraMode, pOption: optino)
        if(result != XSDK_COMPLETE)
        {
            fujiSDK!.xsdk_GetErrorNumber(nil, plAPICode: &apiCode, plERRCode: &errCode)
            CodeNumberToString()
        }
        
        defer
        {
            cameraControlExclusion.unlock()
        }
        return result
    }
    
    func Close(hCamera: XSDK_HANDLE) -> Int
    {
        cameraControlExclusion.lock()
        let result : Int = fujiSDK!.xsdk_Close(hCamera)
        if(result != XSDK_COMPLETE)
        {
            fujiSDK!.xsdk_GetErrorNumber(hCamera, plAPICode: &apiCode, plERRCode: &errCode)
            CodeNumberToString()
        }
        
        defer
        {
            cameraControlExclusion.unlock()
        }
        return result
    }
    
    func GetDeviceInfo(hCamera: XSDK_HANDLE, deviceInfo: inout DeviceInformation) -> Int
    {
        cameraControlExclusion.lock()
        let result : Int = fujiSDK!.xsdk_GetDeviceInfo(hCamera, pDevInfo: deviceInfo)
        if(result != XSDK_COMPLETE)
        {
            fujiSDK!.xsdk_GetErrorNumber(hCamera, plAPICode: &apiCode, plERRCode: &errCode)
            CodeNumberToString()
        }
        
        defer
        {
            cameraControlExclusion.unlock()
        }
        return result
    }
    
    func GetSensitivity(hCamera: XSDK_HANDLE, sensitivity: inout Int) -> Int
    {
        cameraControlExclusion.lock()
        let result : Int = fujiSDK!.xsdk_GetSensitivity(hCamera, plSensitivity: &sensitivity)
        if(result != XSDK_COMPLETE)
        {
            fujiSDK!.xsdk_GetErrorNumber(hCamera, plAPICode: &apiCode, plERRCode: &errCode)
            CodeNumberToString()
        }
        
        defer
        {
            cameraControlExclusion.unlock()
        }
        return result
    }
}
