package com.fujifilm.liveview

import android.app.PendingIntent
import android.content.Intent
import android.hardware.usb.UsbDevice
import android.hardware.usb.UsbDeviceConnection
import android.hardware.usb.UsbManager
import android.os.Build
import androidx.appcompat.app.AppCompatActivity
import android.os.Bundle
import android.widget.AdapterView
import android.widget.ArrayAdapter
import android.widget.Button
import android.widget.CheckBox
import android.widget.ImageView
import android.widget.Spinner

import com.fujifilm.liveview.xsdk.XSDK

class MainActivity : AppCompatActivity() {

    object AndroidSDKResult{
        const val COMPLETE              = 0L
        const val ERROR                 = -1L
    }
    object AndroidSDKErrorNumber{
        const val ERRCODE_NOERR                = 0x00000000L
        const val ERRCODE_SEQUENCE             = 0x00001001L
        const val ERRCODE_PERMISSION           = 0x00001002L
        const val ERRCODE_UNKNOWN              = 0x00009100L
    }

    private lateinit var permissionIntent: PendingIntent
    private lateinit var usbManager: UsbManager
    private var device: UsbDevice? = null
    private var detectedDevice: UsbDevice? = null
    private var usbDeviceConnection: UsbDeviceConnection? = null

    lateinit var checkBox: CheckBox
    lateinit var imageView: ImageView
    private lateinit var button: Button
    private lateinit var spinner: Spinner
    private lateinit var adapter: ArrayAdapter<String>

    private lateinit var liveview:LiveView
    lateinit var errorDialog: ErrorDialog

    val items = mutableListOf<String>()
    private var isRunning = false

    private var detectedcount: Long = 0
    private var errordetails: Long = AndroidSDKErrorNumber.ERRCODE_NOERR

    private val hCamera: XSDK.SDKLong = XSDK.SDKLong(long = 0)

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.activity_main)

        //Preparing the program to obtain permissions
        permissionIntent = PendingIntent.getBroadcast(
            this,
            0,
            Intent("com.fujifilm.liveview.USB_PERMISSION").apply {
                this.`package` = packageName
            },
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) PendingIntent.FLAG_MUTABLE else 0
        )

        //Program initialization process
        if (savedInstanceState == null) {
            usbManager = getSystemService(USB_SERVICE) as UsbManager

            if (usbManager.deviceList.size == 1) {
                device = usbManager.deviceList.values.toList()[0]
            }
        }

        liveview = LiveView(this)
        errorDialog = ErrorDialog(this)

        checkBox = findViewById(R.id.checkBox)
        imageView = findViewById(R.id.imageView)
        button = findViewById(R.id.release_button)
        button.setOnClickListener{
            releaseImage()
        }
        spinner = findViewById(R.id.spinner)

        adapter = ArrayAdapter(this, android.R.layout.simple_spinner_item, items)
        adapter.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item)
        spinner.adapter = adapter

        spinner.onItemSelectedListener = object : AdapterView.OnItemSelectedListener {
            override fun onItemSelected(parent: AdapterView<*>, view: android.view.View, position: Int, id: Long) {
                val selectedItem = items[position]
                val value = CameraControl.ExposureList.find { it.str == selectedItem }
                val numValue = value?.num ?: (selectedItem.toLong())
                val ret = CameraControl.setExposureBias(hCamera.long, numValue)
                if(ret != SDKResult.XSDK_COMPLETE){
                    errorDialog.showErrorDialog(CameraControl.getError(hCamera.long)) {
                    }
                }
            }
            override fun onNothingSelected(parent: AdapterView<*>) {
            }
        }

        initialize()
    }

    override fun onDestroy() {
        super.onDestroy()
        close()
    }

    fun initialize(){
        isRunning = true
        items.clear()
        usbDeviceConnection = null

        var ret = detect()
        if(ret != AndroidSDKResult.COMPLETE){
            isRunning = false
            errorDialog.showErrorDialog("Please allow permissions") {
                // After pressing the OK button, execute the initialization process again.
                initialize()
            }
        }else if(detectedcount == 0L){
            isRunning = false
            errorDialog.showErrorDialog("After connecting the camera to the device,press the OK button") {
                // After pressing the OK button, execute the initialization process again.
                initialize()
            }
        }

        if(isRunning){
            ret = CameraControl.init()
            if(ret != SDKResult.XSDK_COMPLETE){
                isRunning = false
                errorDialog.showErrorDialog("Restart the app") {
                    // After pressing the OK button, execute the initialization process again.
                    initialize()
                }
            }
        }

        if(isRunning){
            ret = open()
            if(ret != AndroidSDKResult.COMPLETE){
                isRunning = false
                if(errordetails == AndroidSDKErrorNumber.ERRCODE_SEQUENCE) {
                    errorDialog.showErrorDialog("After connecting the camera to the device,press the OK button") {
                        // After pressing the OK button, execute the initialization process again.
                        initialize()
                    }
                }else{
                    errorDialog.showErrorDialog(CameraControl.getError(hCamera.long)) {
                        // After pressing the OK button, execute the initialization process again.
                        initialize()
                    }
                }
            }
        }

        if(isRunning){
            CameraControl.setPriorityMode(hCamera.long, 1)
        }

        if(isRunning){
            val apiCode = CameraControl.apiList.find { it.apiName == "API_CODE_SetLiveViewImageQuality" }
            val fine = 0x0001L
            CameraControl.setProp_L(hCamera.long, apiCode!!.apiCode,1,fine)
        }

        if(isRunning){
            val apiCode = CameraControl.apiList.find { it.apiName == "API_CODE_SetLiveViewImageSize" }
            val xga = 0x0001L
            CameraControl.setProp_L(hCamera.long, apiCode!!.apiCode,1,xga)
        }

        if(isRunning){
            val size = XSDK.SDKLong(0)
            val values:MutableList<Long> = mutableListOf()
            ret = CameraControl.capExposureBias(hCamera.long, size, values)
            if(ret == SDKResult.XSDK_COMPLETE){
                for(i in 0 until size.long.toInt()){
                    val num = values[i]
                    val value = CameraControl.ExposureList.find { it.num == num }
                    val numValue = value?.str ?: num.toString()
                    items.add(numValue)
                }
                adapter.notifyDataSetChanged()
            }
        }

        if(isRunning){
            val value = XSDK.SDKLong(0)
            ret = CameraControl.getExposureBias(hCamera.long, value)
            if(ret == SDKResult.XSDK_COMPLETE){
                val selectValue = CameraControl.ExposureList.find { it.num == value.long }
                for(i in 0 until items.size){
                    if(spinner.adapter.getItem(i) == (selectValue?.str ?: value.long.toString())){
                        spinner.setSelection(i)
                        break
                    }
                }
            }
        }

        if(isRunning){
            val slot1_slot2 = 0x0001L
            CameraControl.setMediaRecord(hCamera.long, slot1_slot2)
        }

        if(isRunning){
            liveview.startLiveView(hCamera.long)
        }

        if(isRunning){
            liveview.startGetLiveView(hCamera.long)
        }
    }

    private fun detect():Long{
        var result:Long = AndroidSDKResult.COMPLETE
        errordetails = AndroidSDKErrorNumber.ERRCODE_NOERR

        if (usbManager.deviceList.size == 1) {
            device = usbManager.deviceList.values.toList()[0]
        }else{
            device = null
            detectedDevice = null
            usbDeviceConnection = null
        }
        if (device != null) {
            if (usbManager.hasPermission(device)) {
                detectedDevice = device
                detectedcount = 1
            }else{
                usbManager.requestPermission(device, permissionIntent)
                if (usbManager.hasPermission(device)) {
                    detectedDevice = device
                    detectedcount = 1
                }else{
                    errordetails = AndroidSDKErrorNumber.ERRCODE_PERMISSION
                    result = AndroidSDKResult.ERROR
                }
            }
        }else{
            detectedcount = 0
        }
        return result
    }

    private fun open(): Long {
        errordetails = AndroidSDKErrorNumber.ERRCODE_NOERR

        try {
            if (usbDeviceConnection != null) {
                errordetails = AndroidSDKErrorNumber.ERRCODE_SEQUENCE
                return AndroidSDKResult.ERROR
            }
            if (detectedDevice == null) {
                errordetails = AndroidSDKErrorNumber.ERRCODE_SEQUENCE
                return AndroidSDKResult.ERROR
            }
            val requiredDevice = detectedDevice!!
            if (!usbManager.hasPermission(requiredDevice)) {
                errordetails = AndroidSDKErrorNumber.ERRCODE_PERMISSION
                return AndroidSDKResult.ERROR
            }

            //Open a session and pass the file descriptor to the SDK.
            usbDeviceConnection = usbManager.openDevice(requiredDevice)
            val fd = usbDeviceConnection!!.fileDescriptor
            val descriptors = usbDeviceConnection!!.rawDescriptors
            return CameraControl.setUSBDeviceHandle(fd.toLong(), descriptors, hCamera)
        } catch (e: Exception) {
            errordetails = AndroidSDKErrorNumber.ERRCODE_UNKNOWN
            return AndroidSDKResult.ERROR
        }
    }

    private fun close(){
        liveview.stopLiveView(hCamera.long)
        while (true) {
            val imageInfo = XSDK.XSDK_ImageInformation()
            val ret = CameraControl.readImageInfo(hCamera.long, imageInfo)
            if(ret != SDKResult.XSDK_COMPLETE){
                break
            }
            CameraControl.deleteImage(hCamera.long)
        }
        CameraControl.close(hCamera.long)
        CameraControl.exit()
    }

    private fun releaseImage(){
        val pShortOpt = XSDK.SDKLong(0)
        val pStatus = XSDK.SDKLong(0)
        val ret = CameraControl.release(hCamera.long, Constants.XSDK_RELEASE_EX_S1_ON_S2_ON_S2_OFF_S1_OFF, pShortOpt, pStatus)
        if(ret != SDKResult.XSDK_COMPLETE){
            errorDialog.showErrorDialog(CameraControl.getError(hCamera.long)){
            }
        }
    }
}