package com.fujifilm.liveview

import android.graphics.BitmapFactory
import android.os.Environment
import androidx.lifecycle.lifecycleScope
import com.fujifilm.liveview.xsdk.XSDK
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.io.File
import java.io.FileOutputStream
import java.time.LocalDateTime
import java.time.format.DateTimeFormatter

class LiveView (private val activity: MainActivity){
    private var isStartLive = false

    fun startLiveView(hCamera:Long):Long {
        val apiCode = CameraControl.apiList.find { it.apiName == "API_CODE_StartLiveView" }
        return CameraControl.setProp(hCamera, apiCode!!.apiCode,0)
    }

    fun startGetLiveView(hCamera: Long){
        isStartLive = true
        activity.lifecycleScope.launch {
            withContext(Dispatchers.IO) {
                while (isStartLive){
                    val imageInfo = XSDK.XSDK_ImageInformation()
                    var res = CameraControl.readImageInfo(hCamera, imageInfo)
                    if (res != SDKResult.XSDK_COMPLETE) {
                        val error = CameraControl.getError(hCamera)
                        if (error.contains("XSDK_ERRCODE_COMMUNICATION")) {
                            withContext(Dispatchers.Main){
                                activity.errorDialog.showErrorDialog(error){
                                    activity.initialize()
                                }
                            }
                            isStartLive = false
                            break
                        }
                        continue
                    }
                    when(val type = imageInfo.lFormat.toInt() and 0x1F){
                        Constants.XSDK_IMAGEFORMAT_NONE -> continue
                        Constants.XSDK_IMAGEFORMAT_LIVE -> {
                            val imageData = ByteArray(imageInfo.lDataSize.toInt())
                            res = CameraControl.readImage(hCamera, imageData, imageInfo.lDataSize)
                            if (res != SDKResult.XSDK_COMPLETE) {
                                val error = CameraControl.getError(hCamera)
                                if (error.contains("XSDK_ERRCODE_COMMUNICATION")) {
                                    withContext(Dispatchers.Main){
                                        activity.errorDialog.showErrorDialog(error){
                                            activity.initialize()
                                        }
                                    }
                                    isStartLive = false
                                    break
                                }
                                continue
                            }
                            val imageBitmap = BitmapFactory.decodeByteArray(imageData, 0, imageInfo.lDataSize.toInt())
                            withContext(Dispatchers.Main){
                                activity.imageView.setImageBitmap(imageBitmap)
                            }
                        }
                        else -> {
                            if (activity.checkBox.isChecked) {
                                val imageData = ByteArray(imageInfo.lDataSize.toInt())
                                res = CameraControl.readImage(hCamera, imageData, imageInfo.lDataSize)
                                if (res != SDKResult.XSDK_COMPLETE) {
                                    val error = CameraControl.getError(hCamera)
                                    if (error.contains("XSDK_ERRCODE_COMMUNICATION")) {
                                        withContext(Dispatchers.Main){
                                            activity.errorDialog.showErrorDialog(error){
                                                activity.initialize()
                                            }
                                        }
                                        isStartLive = false
                                        break
                                    }
                                    continue
                                }
                                val externalStorageDir = Environment.getExternalStoragePublicDirectory(Environment.DIRECTORY_PICTURES)
                                val currentDateTime = LocalDateTime.now()
                                val formatter = DateTimeFormatter.ofPattern("yyyyMMdd_HHmmss_SSS")
                                val imageName = when(type){
                                    Constants.XSDK_IMAGEFORMAT_RAW -> {
                                        currentDateTime.format(formatter) + ".RAF"
                                    }
                                    Constants.XSDK_IMAGEFORMAT_JPEG -> {
                                        currentDateTime.format(formatter) + ".JPG"
                                    }
                                    Constants.XSDK_IMAGEFORMAT_HEIF -> {
                                        currentDateTime.format(formatter) + ".HEIC"
                                    }
                                    else -> continue
                                }
                                val folderName = "LiveViewSample"
                                val folder = File(externalStorageDir, folderName)
                                val path = if (!folder.exists()) {
                                    val isCreated = folder.mkdirs()
                                    if (isCreated) {
                                        externalStorageDir.path + "/LiveViewSample"
                                    } else {
                                        externalStorageDir.path
                                    }
                                } else {
                                    externalStorageDir.path + "/LiveViewSample"
                                }

                                val file = File(path, imageName)
                                val fos = FileOutputStream(file)
                                fos.write(imageData)
                                fos.close()
                            }else{
                                CameraControl.deleteImage(hCamera)
                            }
                        }
                    }
                }
            }
        }
    }

    fun stopLiveView(hCamera:Long):Long {
        isStartLive = false
        val apiCode = CameraControl.apiList.find { it.apiName == "API_CODE_StopLiveView" }
        return CameraControl.setProp(hCamera, apiCode!!.apiCode,0)
    }
}